/*
 * Decompiled with CFR 0.152.
 */
package org.apache.brooklyn.util.text;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.annotation.Nonnull;
import org.apache.brooklyn.util.collections.MutableList;
import org.apache.brooklyn.util.exceptions.Exceptions;
import org.apache.brooklyn.util.guava.Maybe;
import org.apache.brooklyn.util.net.URLParamEncoder;
import org.apache.brooklyn.util.text.QuotedStringTokenizer;
import org.apache.brooklyn.util.text.Strings;
import org.apache.brooklyn.util.yaml.Yamls;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StringEscapes {
    private static final Logger log = LoggerFactory.getLogger(StringEscapes.class);

    public static boolean isWrappedInDoubleQuotes(String s) {
        if (Strings.isEmpty(s)) {
            return false;
        }
        if (!s.startsWith("\"") || !s.endsWith("\"")) {
            return false;
        }
        return s.substring(1, s.length() - 1).replace("\\\\", "").replace("\\\"", "").indexOf("\"") == -1;
    }

    public static boolean isWrappedInSingleQuotes(String s) {
        if (Strings.isEmpty(s)) {
            return false;
        }
        if (!s.startsWith("'") || !s.endsWith("'")) {
            return false;
        }
        return s.substring(1, s.length() - 1).replace("\\\\", "").replace("\\'", "").indexOf("'") == -1;
    }

    public static boolean isWrappedInMatchingQuotes(String s) {
        return StringEscapes.isWrappedInDoubleQuotes(s) || StringEscapes.isWrappedInSingleQuotes(s);
    }

    public static String escapeUrlParam(String input) {
        return URLParamEncoder.encode(input);
    }

    public static String escapeHtmlFormUrl(String url) {
        try {
            return URLEncoder.encode(url, "UTF-8");
        }
        catch (UnsupportedEncodingException e) {
            throw Exceptions.propagate(e);
        }
    }

    public static String escapeSql(String x) {
        if (x == null) {
            return null;
        }
        return x.replaceAll("'", "''");
    }

    private static void appendEscaped(Appendable out, char c) throws IOException {
        out.append('\\');
        out.append(c);
    }

    private static String applyUnquoteAndUnescape(String s, String mode, boolean allowMultipleQuotes) {
        StringBuilder result = new StringBuilder();
        boolean escaped = false;
        boolean quoted = false;
        for (int i = 0; i < s.length(); ++i) {
            char c = s.charAt(i);
            if (!quoted) {
                assert (i == 0 || allowMultipleQuotes);
                assert (!escaped);
                if (c == '\"') {
                    quoted = true;
                    continue;
                }
                if (!allowMultipleQuotes) {
                    throw new IllegalArgumentException("String '" + s + "' is not a valid " + mode + " string (must start with double quote)");
                }
                result.append(c);
                continue;
            }
            if (escaped) {
                if (c == '\\' || c == '\"' || c == '\'') {
                    result.append(c);
                } else if (c == 'n') {
                    result.append('\n');
                } else if (c == 't') {
                    result.append('\t');
                } else if (c == 'r') {
                    result.append('\r');
                } else {
                    throw new IllegalArgumentException("String '" + s + "' is not a valid " + mode + " string (unsupported escape char '" + c + "' at position " + i + ")");
                }
                escaped = false;
                continue;
            }
            if (c == '\\') {
                escaped = true;
                continue;
            }
            if (c == '\"') {
                quoted = false;
                if (allowMultipleQuotes || i >= s.length() - 1) continue;
                throw new IllegalArgumentException("String '" + s + "' is not a valid " + mode + " string (unescaped interior double quote at position " + i + ")");
            }
            result.append(c);
        }
        if (quoted) {
            throw new IllegalArgumentException("String '" + s + "' is not a valid " + mode + " string (unterminated string)");
        }
        assert (!escaped);
        return result.toString();
    }

    public static class JavaStringEscapes {
        public static String escapeJavaString(String value) {
            StringBuilder out = new StringBuilder();
            try {
                JavaStringEscapes.escapeJavaString(value, out);
            }
            catch (IOException e) {
                throw Exceptions.propagate(e);
            }
            return out.toString();
        }

        public static String wrapJavaString(String value) {
            StringBuilder out = new StringBuilder();
            try {
                JavaStringEscapes.wrapJavaString(value, out);
            }
            catch (IOException e) {
                throw Exceptions.propagate(e);
            }
            return out.toString();
        }

        public static List<String> wrapJavaStrings(Iterable<String> values) {
            if (values == null) {
                return null;
            }
            MutableList<String> result = MutableList.of();
            for (String v : values) {
                result.add(JavaStringEscapes.wrapJavaString(v));
            }
            return result;
        }

        public static String unwrapJavaStringIfWrapped(String s) {
            if (!StringEscapes.isWrappedInDoubleQuotes(s)) {
                return s;
            }
            return JavaStringEscapes.unwrapJavaString(s);
        }

        public static void wrapJavaString(String value, Appendable out) throws IOException {
            if (value == null) {
                out.append("null");
            } else {
                out.append('\"');
                JavaStringEscapes.escapeJavaString(value, out);
                out.append('\"');
            }
        }

        public static void escapeJavaString(@Nonnull String value, Appendable out) throws IOException {
            for (int i = 0; i < value.length(); ++i) {
                char c = value.charAt(i);
                if (c == '\\' || c == '\"') {
                    StringEscapes.appendEscaped(out, c);
                    continue;
                }
                if (c == '\n') {
                    StringEscapes.appendEscaped(out, 'n');
                    continue;
                }
                if (c == '\t') {
                    StringEscapes.appendEscaped(out, 't');
                    continue;
                }
                if (c == '\r') {
                    StringEscapes.appendEscaped(out, 'r');
                    continue;
                }
                out.append(c);
            }
        }

        public static String unwrapJavaString(String s) {
            return StringEscapes.applyUnquoteAndUnescape(s, "Java", false);
        }

        public static List<String> unwrapQuotedJavaStringList(String s, String separator) {
            ArrayList<String> result = new ArrayList<String>();
            String remaining = s.trim();
            while (remaining.length() > 0) {
                int endIndex = JavaStringEscapes.findNextClosingQuoteOf(remaining);
                result.add(JavaStringEscapes.unwrapJavaString(remaining.substring(0, endIndex + 1)));
                remaining = remaining.substring(endIndex + 1).trim();
                if (remaining.startsWith(separator)) {
                    remaining = remaining.substring(separator.length()).trim();
                    continue;
                }
                if (remaining.length() <= 0) continue;
                throw new IllegalArgumentException("String '" + s + "' has invalid separators, should be '" + separator + "'");
            }
            return result;
        }

        private static int findNextClosingQuoteOf(String s) {
            boolean escaped = false;
            boolean quoted = false;
            for (int i = 0; i < s.length(); ++i) {
                char c = s.charAt(i);
                if (!quoted) {
                    assert (i == 0);
                    assert (!escaped);
                    if (c == '\"') {
                        quoted = true;
                        continue;
                    }
                    throw new IllegalArgumentException("String '" + s + "' is not a valid Java string (must start with double quote)");
                }
                if (escaped) {
                    escaped = false;
                    continue;
                }
                if (c == '\\') {
                    escaped = true;
                    continue;
                }
                if (c != '\"') continue;
                quoted = false;
                return i;
            }
            assert (quoted);
            throw new IllegalArgumentException("String '" + s + "' is not a valid Java string (unterminated string)");
        }

        public static List<String> unwrapJsonishListIfPossible(String input) {
            return JavaStringEscapes.unwrapJsonishListStringIfPossible(input);
        }

        public static Maybe<List<Object>> tryUnwrapJsonishList(String input) {
            Matcher m;
            if (input == null) {
                return Maybe.absent("null input cannot unwrap to a list");
            }
            String inputT = input.trim();
            String inputYaml = null;
            if (!inputT.startsWith("[") && !inputT.endsWith("]")) {
                inputYaml = "[" + inputT + "]";
            }
            if (inputT.startsWith("[") && inputT.endsWith("]")) {
                inputYaml = inputT;
            }
            if (inputYaml != null) {
                try {
                    Object r = Iterables.getOnlyElement(Yamls.parseAll(inputYaml));
                    if (r instanceof List) {
                        List result = (List)r;
                        return Maybe.of(result);
                    }
                }
                catch (Exception e) {
                    Exceptions.propagateIfFatal(e);
                }
                if (inputT.startsWith("[")) {
                    return Maybe.absent("Supplied format looked like YAML but could not parse as YAML");
                }
            }
            MutableList result = MutableList.of();
            Pattern dq = Pattern.compile("^\\s*(\"([^\"\\\\]|[\\\\.])*\")\\s*(,|$)");
            Pattern nq = Pattern.compile("^\\s*(|[^,\"\\s]([^,\"]*[^,\"\\s])?)\\s*(,|$)");
            int removedChars = 0;
            do {
                Object ri;
                if ((m = dq.matcher(input)).find()) {
                    try {
                        ri = JavaStringEscapes.unwrapJavaString(m.group(1));
                    }
                    catch (Exception e) {
                        Exceptions.propagateIfFatal(e);
                        return Maybe.absent("Could not match valid quote pattern" + (removedChars > 0 ? " at position " + removedChars : "") + ": " + Exceptions.collapseText(e));
                    }
                } else {
                    m = nq.matcher(input);
                    if (m.find()) {
                        String w = m.group(1);
                        ri = w;
                        if (w.matches("[0-9]*\\.[0-9]+")) {
                            try {
                                ri = Double.parseDouble(w);
                            }
                            catch (Exception exception) {}
                        } else if (w.matches("[0-9]+")) {
                            try {
                                ri = Integer.parseInt(w);
                            }
                            catch (Exception exception) {}
                        } else if (Boolean.TRUE.toString().equals(w)) {
                            ri = true;
                        } else if (Boolean.FALSE.toString().equals(w)) {
                            ri = false;
                        } else if ("null".equals(w)) {
                            ri = null;
                        }
                    } else {
                        return Maybe.absent("Could not match valid quote pattern" + (removedChars > 0 ? " at position " + removedChars : ""));
                    }
                }
                result.add(ri);
                input = input.substring(m.end());
                removedChars += m.end();
            } while (m.group(0).endsWith(","));
            return Maybe.of(result);
        }

        public static List<String> unwrapJsonishListStringIfPossible(String input) {
            try {
                return JavaStringEscapes.unwrapOptionallyQuotedJavaStringList(input);
            }
            catch (Exception e) {
                Exceptions.propagateIfFatal(e);
                if (e instanceof IllegalArgumentException) {
                    if (log.isDebugEnabled()) {
                        log.debug("Unable to parse JSON list '" + input + "' (" + e + "); treating as single-element string list");
                    }
                } else {
                    log.warn("Unable to parse JSON list '" + input + "' (" + e + "); treating as single-element string list", (Throwable)e);
                }
                return MutableList.of(input);
            }
        }

        public static List<String> unwrapOptionallyQuotedJavaStringList(String input) {
            boolean inBrackets;
            if (input == null) {
                return null;
            }
            MutableList<String> result = MutableList.of();
            String i1 = input.trim();
            boolean bl = inBrackets = i1.startsWith("[") && i1.endsWith("]");
            if (inBrackets) {
                i1 = i1.substring(1, i1.length() - 1).trim();
            }
            QuotedStringTokenizer qst = new QuotedStringTokenizer(i1, "\"", true, ",", false);
            while (qst.hasMoreTokens()) {
                String t = qst.nextToken().trim();
                if (StringEscapes.isWrappedInDoubleQuotes(t)) {
                    result.add(JavaStringEscapes.unwrapJavaString(t));
                    continue;
                }
                if (inBrackets && (t.indexOf(91) >= 0 || t.indexOf(93) >= 0)) {
                    throw new IllegalArgumentException("Literal square brackets must be quoted, in element '" + t + "'");
                }
                result.add(t.trim());
            }
            return result;
        }
    }

    public static class BashStringEscapes {
        private static final String BACKSLASH = "\\\\";
        private static final String DOUBLE_QUOTE = "\\\"";

        public static Function<String, String> wrapBash() {
            return new WrapBashFunction();
        }

        public static String wrapBash(String value) {
            StringBuilder out = new StringBuilder();
            try {
                BashStringEscapes.wrapBash(value, out);
            }
            catch (IOException e) {
                throw Exceptions.propagate(e);
            }
            return out.toString();
        }

        public static void wrapBash(String value, Appendable out) throws IOException {
            out.append('\"');
            BashStringEscapes.escapeLiteralForDoubleQuotedBash(value, out);
            out.append('\"');
        }

        private static void escapeLiteralForDoubleQuotedBash(String value, Appendable out) throws IOException {
            for (int i = 0; i < value.length(); ++i) {
                char c = value.charAt(i);
                if (c == '\\' || c == '\"' || c == '$' || c == '`') {
                    StringEscapes.appendEscaped(out, c);
                    continue;
                }
                if (c == '!') {
                    out.append("\"'!'\"");
                    continue;
                }
                out.append(c);
            }
        }

        public static String escapeLiteralForDoubleQuotedBash(String unquotedInputToBeEscaped) {
            StringBuilder out = new StringBuilder();
            try {
                BashStringEscapes.escapeLiteralForDoubleQuotedBash(unquotedInputToBeEscaped, out);
            }
            catch (IOException e) {
                throw Exceptions.propagate(e);
            }
            return out.toString();
        }

        public static String doubleQuoteLiteralsForBash(String ... args) {
            StringBuilder result = new StringBuilder();
            for (String arg : args) {
                if (!Strings.isEmpty(result)) {
                    result.append(" ");
                }
                result.append("\"");
                result.append(BashStringEscapes.escapeLiteralForDoubleQuotedBash(arg));
                result.append("\"");
            }
            return result.toString();
        }

        public static boolean isValidForDoubleQuotingInBash(String x) {
            return BashStringEscapes.checkValidForDoubleQuotingInBash(x) == null;
        }

        public static void assertValidForDoubleQuotingInBash(String x) {
            String problem = BashStringEscapes.checkValidForDoubleQuotingInBash(x);
            if (problem == null) {
                return;
            }
            throw new IllegalArgumentException("String \"" + x + "\" not acceptable for bash argument (including double quotes): " + problem);
        }

        private static String checkValidForDoubleQuotingInBash(String x) {
            if (x.matches("[^\\\\\\\"]*(\\\\\\\\)*\\\".*")) {
                return "unescaped double quote";
            }
            return null;
        }

        public static String unwrapBashQuotesAndEscapes(String s) {
            return StringEscapes.applyUnquoteAndUnescape(s, "Bash", true);
        }

        public static class WrapBashFunction
        implements Function<String, String> {
            public String apply(String input) {
                return BashStringEscapes.wrapBash(input);
            }
        }
    }
}

