/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.compress.compressors.gzip;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.zip.GZIPInputStream;
import java.util.zip.ZipException;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.compress.compressors.gzip.ExtraField;
import org.apache.commons.compress.compressors.gzip.GzipCompressorInputStream;
import org.apache.commons.compress.compressors.gzip.GzipCompressorOutputStream;
import org.apache.commons.compress.compressors.gzip.GzipParameters;
import org.apache.commons.lang3.ArrayFill;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;
import shaded.org.apache.commons.io.IOUtils;

class GzipCompressorOutputStreamTest {
    private static final String EXPECTED_BASE_NAME = "\u6d4b\u8bd5\u4e2d\u6587\u540d\u79f0";
    private static final String EXPECTED_FILE_NAME = "\u6d4b\u8bd5\u4e2d\u6587\u540d\u79f0.xml";

    GzipCompressorOutputStreamTest() {
    }

    private void testChineseFileName(String expected, String sourceFile, Charset fileNameCharset) throws IOException {
        String unicodeFileName;
        byte[] fileNameBytes;
        Path tempSourceFile = Files.createTempFile(sourceFile, sourceFile, new FileAttribute[0]);
        byte[] bytes = "<text>Hello World!</text>".getBytes(StandardCharsets.ISO_8859_1);
        Files.write(tempSourceFile, bytes, new OpenOption[0]);
        Path targetFile = Files.createTempFile(EXPECTED_BASE_NAME, ".gz", new FileAttribute[0]);
        GzipParameters parameters = new GzipParameters();
        parameters.setFileNameCharset(fileNameCharset);
        Assertions.assertEquals((Object)fileNameCharset, (Object)parameters.getFileNameCharset());
        parameters.setFileName(EXPECTED_FILE_NAME);
        parameters.setComment("Comment on \u6d4b\u8bd5\u4e2d\u6587\u540d\u79f0.xml");
        try (OutputStream fos = Files.newOutputStream(targetFile, new OpenOption[0]);
             GzipCompressorOutputStream gos = new GzipCompressorOutputStream(fos, parameters);){
            gos.write(tempSourceFile);
        }
        try (GzipCompressorInputStream gis = new GzipCompressorInputStream(Files.newInputStream(targetFile, new OpenOption[0]));){
            fileNameBytes = gis.getMetaData().getFileName().getBytes(StandardCharsets.ISO_8859_1);
            unicodeFileName = new String(fileNameBytes, fileNameCharset);
            Assertions.assertEquals((Object)expected, (Object)unicodeFileName);
            Assertions.assertArrayEquals((byte[])bytes, (byte[])IOUtils.toByteArray((InputStream)gis));
        }
        gis = ((GzipCompressorInputStream.Builder)GzipCompressorInputStream.builder().setPath(targetFile)).setFileNameCharset(fileNameCharset).get();
        try {
            fileNameBytes = gis.getMetaData().getFileName().getBytes(fileNameCharset);
            unicodeFileName = new String(fileNameBytes, fileNameCharset);
            Assertions.assertEquals((Object)expected, (Object)unicodeFileName);
            Assertions.assertArrayEquals((byte[])bytes, (byte[])IOUtils.toByteArray((InputStream)gis));
            gis.getMetaData().setTrailerCrc(0L);
            gis.getMetaData().setTrailerISize(0L);
            Assertions.assertEquals((Object)parameters, (Object)gis.getMetaData());
        }
        finally {
            if (gis != null) {
                gis.close();
            }
        }
    }

    @Test
    void testChineseFileNameGBK() throws IOException {
        Assumptions.assumeTrue((boolean)Charset.isSupported("GBK"));
        this.testChineseFileName(EXPECTED_FILE_NAME, EXPECTED_FILE_NAME, Charset.forName("GBK"));
    }

    @Test
    void testChineseFileNameUTF8() throws IOException {
        this.testChineseFileName(EXPECTED_FILE_NAME, EXPECTED_FILE_NAME, StandardCharsets.UTF_8);
    }

    @ParameterizedTest
    @CsvSource(value={"0,    42, false", "1,      , true", "1,     0, false", "1, 65531, false", "1, 65532, true", "2,     0, false", "2, 32764, true", "2, 32763, false"})
    void testExtraSubfields(int subFieldCount, Integer payloadSize, boolean shouldFail) throws IOException {
        Path tempSourceFile = Files.createTempFile("test_gzip_extra_", ".txt", new FileAttribute[0]);
        Path targetFile = Files.createTempFile("test_gzip_extra_", ".txt.gz", new FileAttribute[0]);
        Files.write(tempSourceFile, "Hello World!".getBytes(StandardCharsets.ISO_8859_1), new OpenOption[0]);
        GzipParameters parameters = new GzipParameters();
        ExtraField extra = new ExtraField();
        boolean failed = false;
        byte[][] payloads = new byte[subFieldCount][];
        for (int i = 0; i < subFieldCount; ++i) {
            if (payloadSize != null) {
                payloads[i] = ArrayFill.fill((byte[])new byte[payloadSize.intValue()], (byte)((byte)(97 + i)));
            }
            try {
                extra.addSubField("z" + i, payloads[i]);
                continue;
            }
            catch (IOException | NullPointerException e) {
                failed = true;
                break;
            }
        }
        Assertions.assertEquals((Object)shouldFail, (Object)failed, (String)("add subfield " + (shouldFail ? "succes" : "failure") + " was not expected."));
        if (shouldFail) {
            return;
        }
        if (subFieldCount > 0) {
            Assertions.assertThrows(UnsupportedOperationException.class, () -> extra.iterator().remove());
        }
        parameters.setExtraField(extra);
        try (OutputStream fos = Files.newOutputStream(targetFile, new OpenOption[0]);
             GzipCompressorOutputStream gos = new GzipCompressorOutputStream(fos, parameters);){
            gos.write(tempSourceFile);
            gos.close();
            Assertions.assertTrue((boolean)gos.isClosed());
        }
        try (GzipCompressorInputStream gis = new GzipCompressorInputStream(Files.newInputStream(targetFile, new OpenOption[0]));){
            ExtraField extra2 = gis.getMetaData().getExtraField();
            Assertions.assertEquals((Object)parameters, (Object)gis.getMetaData());
            Assertions.assertEquals((Object)(subFieldCount == 0 ? 1 : 0), (Object)extra2.isEmpty());
            Assertions.assertEquals((int)subFieldCount, (int)extra2.size());
            Assertions.assertEquals((int)(4 * subFieldCount + subFieldCount * payloadSize), (int)extra2.getEncodedSize());
            ArrayList listCopy = new ArrayList();
            extra2.forEach(listCopy::add);
            Assertions.assertEquals((int)subFieldCount, (int)listCopy.size());
            for (int i = 0; i < subFieldCount; ++i) {
                ExtraField.SubField sf = extra2.getSubField(i);
                Assertions.assertSame((Object)sf, listCopy.get(i));
                Assertions.assertSame((Object)sf, (Object)extra2.findFirstSubField("z" + i));
                Assertions.assertEquals((Object)("z" + i), (Object)sf.getId());
                Assertions.assertArrayEquals((byte[])payloads[i], (byte[])sf.getPayload(), (String)("field " + i + " has wrong payload"));
            }
            extra2.clear();
            Assertions.assertTrue((boolean)extra2.isEmpty());
        }
    }

    @Test
    void testExtraSubfieldsEmpty() {
        ExtraField extra = new ExtraField();
        Assertions.assertEquals((int)0, (int)extra.toByteArray().length);
        Assertions.assertFalse((boolean)extra.iterator().hasNext());
        extra.forEach(e -> Assertions.fail((String)"Not empty."));
        Assertions.assertThrows(IndexOutOfBoundsException.class, () -> extra.getSubField(0));
    }

    private void testFileName(String expected, String sourceFile) throws IOException {
        Path tempSourceFile = Files.createTempFile(sourceFile, sourceFile, new FileAttribute[0]);
        byte[] bytes = "<text>Hello World!</text>".getBytes(StandardCharsets.ISO_8859_1);
        Files.write(tempSourceFile, bytes, new OpenOption[0]);
        Path targetFile = Files.createTempFile("test", ".gz", new FileAttribute[0]);
        GzipParameters parameters = new GzipParameters();
        parameters.setFilename(sourceFile);
        Assertions.assertEquals((Object)parameters.getFilename(), (Object)parameters.getFileName());
        parameters.setFileName(sourceFile);
        Assertions.assertEquals((Object)parameters.getFilename(), (Object)parameters.getFileName());
        try (OutputStream fos = Files.newOutputStream(targetFile, new OpenOption[0]);
             GzipCompressorOutputStream gos = new GzipCompressorOutputStream(fos, parameters);){
            gos.write(tempSourceFile);
        }
        try (GzipCompressorInputStream gis = new GzipCompressorInputStream(Files.newInputStream(targetFile, new OpenOption[0]));){
            Assertions.assertEquals((Object)expected, (Object)gis.getMetaData().getFileName());
            Assertions.assertEquals((Object)expected, (Object)gis.getMetaData().getFilename());
            Assertions.assertArrayEquals((byte[])bytes, (byte[])IOUtils.toByteArray((InputStream)gis));
        }
    }

    @Test
    void testFileNameAscii() throws IOException {
        this.testFileName("ASCII.xml", "ASCII.xml");
    }

    @Test
    void testFileNameChinesePercentEncoded() throws IOException {
        this.testFileName("??????.xml", EXPECTED_FILE_NAME);
    }

    @Test
    void testHeaderCrc() throws IOException, DecoderException {
        GzipParameters parameters = new GzipParameters();
        parameters.setHeaderCRC(true);
        parameters.setModificationTime(1716864051L);
        parameters.setFileName("AAAA");
        parameters.setComment("ZZZZ");
        parameters.setOS(GzipParameters.OS.UNIX);
        ExtraField extra = new ExtraField();
        extra.addSubField("BB", "CCCC".getBytes(StandardCharsets.ISO_8859_1));
        parameters.setExtraField(extra);
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        GzipCompressorOutputStream gos = new GzipCompressorOutputStream((OutputStream)baos, parameters);
        gos.close();
        byte[] result = baos.toByteArray();
        byte[] expected = Hex.decodeHex((String)"1f8b081e3344556600030800424204004343434341414141005a5a5a5a00d84203000000000000000000");
        Assertions.assertArrayEquals((byte[])expected, (byte[])result);
        Assertions.assertDoesNotThrow(() -> {
            GZIPInputStream gis = new GZIPInputStream(new ByteArrayInputStream(result));
            gis.close();
        });
        try (GzipCompressorInputStream gis = new GzipCompressorInputStream((InputStream)new ByteArrayInputStream(result));){
            GzipParameters metaData = gis.getMetaData();
            Assertions.assertTrue((boolean)metaData.getHeaderCRC());
            Assertions.assertEquals((long)1716864051L, (long)metaData.getModificationTime());
            Assertions.assertEquals((int)1, (int)metaData.getExtraField().size());
            ExtraField.SubField sf = (ExtraField.SubField)metaData.getExtraField().iterator().next();
            Assertions.assertEquals((Object)"BB", (Object)sf.getId());
            Assertions.assertEquals((Object)"CCCC", (Object)new String(sf.getPayload(), StandardCharsets.ISO_8859_1));
            Assertions.assertEquals((Object)"AAAA", (Object)metaData.getFileName());
            Assertions.assertEquals((Object)"ZZZZ", (Object)metaData.getComment());
            Assertions.assertEquals((Object)GzipParameters.OS.UNIX, (Object)metaData.getOS());
            Assertions.assertEquals((Object)parameters, (Object)metaData);
        }
        Assertions.assertThrows(ZipException.class, () -> {
            result[30] = 119;
            GZIPInputStream gis = new GZIPInputStream(new ByteArrayInputStream(result));
            gis.close();
        }, (String)"Header CRC verification is no longer feasible with JDK classes. The earlier assertion would have passed despite a bad header CRC.");
    }
}

