/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.vfs2.impl;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.ArrayDeque;
import java.util.Deque;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.vfs2.FileChangeEvent;
import org.apache.commons.vfs2.FileListener;
import org.apache.commons.vfs2.FileObject;
import org.apache.commons.vfs2.FileSystemManager;
import org.apache.commons.vfs2.VFS;
import org.apache.commons.vfs2.VfsTestUtils;
import org.apache.commons.vfs2.impl.DefaultFileMonitor;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;

public class DefaultFileMonitorTest {
    private static final int DELAY_MILLIS = 100;
    private FileSystemManager fileSystemManager;
    private final Deque<Status> status = new ArrayDeque<Status>();
    private File testDir;
    private File testFile;

    private void deleteTestFileIfPresent() {
        if (this.testFile != null && this.testFile.exists()) {
            boolean deleted = this.testFile.delete();
            Assertions.assertTrue((boolean)deleted, (String)this.testFile.toString());
        }
    }

    private Status getStatus(PeekLocation peekLocation) {
        switch (Objects.requireNonNull(peekLocation, "peekLocation").ordinal()) {
            case 0: {
                return this.status.peekFirst();
            }
            case 1: {
                return this.status.peekLast();
            }
        }
        throw new IllegalStateException();
    }

    private void resetStatus() {
        this.status.clear();
    }

    @BeforeEach
    public void setUp() throws Exception {
        this.fileSystemManager = VFS.getManager();
        this.testDir = VfsTestUtils.getTestDirectoryFile();
        this.resetStatus();
        this.testFile = new File(this.testDir, "testReload.properties");
        this.deleteTestFileIfPresent();
    }

    @AfterEach
    public void tearDown() {
        this.deleteTestFileIfPresent();
    }

    @Test
    public void testChildFileDeletedWithoutRecursiveChecking() throws Exception {
        this.writeToFile(this.testFile);
        try (FileObject fileObject = this.fileSystemManager.resolveFile(this.testDir.toURI().toURL().toString());
             DefaultFileMonitor monitor = new DefaultFileMonitor((FileListener)new TestFileListener());){
            monitor.setDelay(2000L);
            monitor.setRecursive(false);
            monitor.addFile(fileObject);
            monitor.start();
            this.resetStatus();
            Thread.sleep(500L);
            this.testFile.delete();
            Thread.sleep(3000L);
            Assertions.assertNull((Object)((Object)this.getStatus(PeekLocation.LAST)), (String)"Event should not have occurred");
        }
    }

    @Test
    public void testChildFileRecreated() throws Exception {
        this.writeToFile(this.testFile);
        try (FileObject fileObj = this.fileSystemManager.resolveFile(this.testDir.toURI().toURL().toString());
             DefaultFileMonitor monitor = new DefaultFileMonitor((FileListener)new TestFileListener());){
            monitor.setDelay(2000L);
            monitor.setRecursive(true);
            monitor.addFile(fileObj);
            monitor.start();
            this.resetStatus();
            Thread.sleep(500L);
            this.testFile.delete();
            this.waitFor(Status.DELETED, 3000L, PeekLocation.LAST);
            this.resetStatus();
            Thread.sleep(500L);
            this.writeToFile(this.testFile);
            this.waitFor(Status.CREATED, 3000L, PeekLocation.LAST);
        }
    }

    @Test
    public void testFileCreated() throws Exception {
        try (FileObject fileObject = this.fileSystemManager.resolveFile(this.testFile.toURI().toURL().toString());
             DefaultFileMonitor monitor = new DefaultFileMonitor((FileListener)new TestFileListener());){
            monitor.setDelay(100L);
            monitor.addFile(fileObject);
            monitor.start();
            this.writeToFile(this.testFile);
            Thread.sleep(500L);
            this.waitFor(Status.CREATED, 500L, PeekLocation.FIRST);
        }
    }

    @Test
    public void testFileDeleted() throws Exception {
        this.writeToFile(this.testFile);
        try (FileObject fileObject = this.fileSystemManager.resolveFile(this.testFile.toURI().toString());
             DefaultFileMonitor monitor = new DefaultFileMonitor((FileListener)new TestFileListener());){
            monitor.setDelay(100L);
            monitor.addFile(fileObject);
            monitor.start();
            this.testFile.delete();
            this.waitFor(Status.DELETED, 500L, PeekLocation.LAST);
        }
    }

    @Test
    public void testFileModified() throws Exception {
        this.writeToFile(this.testFile);
        try (FileObject fileObject = this.fileSystemManager.resolveFile(this.testFile.toURI().toURL().toString());
             DefaultFileMonitor monitor = new DefaultFileMonitor((FileListener)new TestFileListener());){
            monitor.setDelay(100L);
            monitor.addFile(fileObject);
            monitor.start();
            Thread.sleep(1000L);
            long valueMillis = System.currentTimeMillis();
            boolean rcMillis = this.testFile.setLastModified(valueMillis);
            Assertions.assertTrue((boolean)rcMillis, (String)"setLastModified succeeded");
            this.waitFor(Status.CHANGED, 500L, PeekLocation.LAST);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Test
    public void testFileMonitorRestarted() throws Exception {
        try (FileObject fileObject = this.fileSystemManager.resolveFile(this.testFile.toURI().toString());){
            DefaultFileMonitor monitor = new DefaultFileMonitor((FileListener)new TestFileListener());
            try {
                monitor.setDelay(100L);
                monitor.addFile(fileObject);
                monitor.start();
                this.writeToFile(this.testFile);
                Thread.sleep(500L);
            }
            finally {
                monitor.stop();
            }
            monitor.start();
            try {
                this.testFile.delete();
                this.waitFor(Status.DELETED, 500L, PeekLocation.LAST);
            }
            finally {
                monitor.stop();
            }
        }
    }

    @Test
    public void testFileRecreated() throws Exception {
        try (FileObject fileObject = this.fileSystemManager.resolveFile(this.testFile.toURI());
             DefaultFileMonitor monitor = new DefaultFileMonitor((FileListener)new TestFileListener());){
            monitor.setDelay(100L);
            monitor.addFile(fileObject);
            monitor.start();
            this.writeToFile(this.testFile);
            this.waitFor(Status.CREATED, 1000L, PeekLocation.LAST);
            this.resetStatus();
            this.testFile.delete();
            this.waitFor(Status.DELETED, 1000L, PeekLocation.LAST);
            this.resetStatus();
            Thread.sleep(500L);
            monitor.addFile(fileObject);
            this.writeToFile(this.testFile);
            this.waitFor(Status.CREATED, 1000L, PeekLocation.LAST);
        }
    }

    @Disabled(value="VFS-299")
    @Test
    public void testIgnoreTestAddRemove() throws Exception {
        try (FileObject fileObject = this.fileSystemManager.resolveFile(this.testFile.toURI().toString());){
            CountingListener listener = new CountingListener();
            try (DefaultFileMonitor monitor = new DefaultFileMonitor((FileListener)listener);){
                monitor.setDelay(100L);
                monitor.addFile(fileObject);
                monitor.removeFile(fileObject);
                monitor.addFile(fileObject);
                monitor.start();
                this.writeToFile(this.testFile);
                Thread.sleep(300L);
                Assertions.assertEquals((long)1L, (long)listener.created.get(), (String)"Created event is only fired once");
            }
        }
    }

    @Disabled(value="VFS-299")
    @Test
    public void testIgnoreTestStartStop() throws Exception {
        try (FileObject fileObject = this.fileSystemManager.resolveFile(this.testFile.toURI().toString());){
            CountingListener stoppedListener = new CountingListener();
            try (DefaultFileMonitor stoppedMonitor = new DefaultFileMonitor((FileListener)stoppedListener);){
                stoppedMonitor.start();
                stoppedMonitor.addFile(fileObject);
            }
            CountingListener activeListener = new CountingListener();
            try (DefaultFileMonitor activeMonitor = new DefaultFileMonitor((FileListener)activeListener);){
                activeMonitor.setDelay(100L);
                activeMonitor.addFile(fileObject);
                activeMonitor.start();
                this.writeToFile(this.testFile);
                Thread.sleep(1000L);
                Assertions.assertEquals((long)1L, (long)activeListener.created.get(), (String)"The listener of the active monitor received one created event");
                Assertions.assertEquals((long)0L, (long)stoppedListener.created.get(), (String)"The listener of the stopped monitor received no events");
            }
        }
    }

    private void waitFor(Status expected, long timeoutMillis, PeekLocation peekLocation) throws InterruptedException {
        if (expected == this.getStatus(peekLocation)) {
            return;
        }
        long interval = timeoutMillis / 10L;
        for (long remaining = timeoutMillis; remaining > 0L; remaining -= interval) {
            Thread.sleep(interval);
            if (expected != this.getStatus(peekLocation)) continue;
            return;
        }
        Assertions.assertNotNull((Object)((Object)this.getStatus(peekLocation)), (String)"No event occurred");
        Assertions.assertEquals((Object)((Object)expected), (Object)((Object)this.getStatus(peekLocation)), (String)("Incorrect event " + (Object)((Object)this.getStatus(peekLocation))));
    }

    private void writeToFile(File file) throws IOException {
        Files.write(file.toPath(), "string=value1".getBytes(StandardCharsets.UTF_8), new OpenOption[0]);
    }

    private static enum PeekLocation {
        FIRST,
        LAST;

    }

    private static enum Status {
        CHANGED,
        CREATED,
        DELETED;

    }

    private class TestFileListener
    implements FileListener {
        private TestFileListener() {
        }

        public void fileChanged(FileChangeEvent event) throws Exception {
            DefaultFileMonitorTest.this.status.add(Status.CHANGED);
        }

        public void fileCreated(FileChangeEvent event) throws Exception {
            DefaultFileMonitorTest.this.status.add(Status.CREATED);
        }

        public void fileDeleted(FileChangeEvent event) throws Exception {
            DefaultFileMonitorTest.this.status.add(Status.DELETED);
        }
    }

    private static class CountingListener
    implements FileListener {
        private final AtomicLong changed = new AtomicLong();
        private final AtomicLong created = new AtomicLong();
        private final AtomicLong deleted = new AtomicLong();

        private CountingListener() {
        }

        public void fileChanged(FileChangeEvent event) {
            this.changed.incrementAndGet();
        }

        public void fileCreated(FileChangeEvent event) {
            this.created.incrementAndGet();
        }

        public void fileDeleted(FileChangeEvent event) {
            this.deleted.incrementAndGet();
        }
    }
}

