﻿using J2N.Text;
using Lucene.Net.Attributes;
using Lucene.Net.Documents;
using Lucene.Net.Index.Extensions;
using NUnit.Framework;
using RandomizedTesting.Generators;
using System;
using System.Text.RegularExpressions;

namespace Lucene.Net.Index
{
    /*
     * Licensed to the Apache Software Foundation (ASF) under one or more
     * contributor license agreements.  See the NOTICE file distributed with
     * this work for additional information regarding copyright ownership.
     * The ASF licenses this file to You under the Apache License, Version 2.0
     * (the "License"); you may not use this file except in compliance with
     * the License.  You may obtain a copy of the License at
     *
     *     http://www.apache.org/licenses/LICENSE-2.0
     *
     * Unless required by applicable law or agreed to in writing, software
     * distributed under the License is distributed on an "AS IS" BASIS,
     * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
     * See the License for the specific language governing permissions and
     * limitations under the License.
     */

    using BytesRef = Lucene.Net.Util.BytesRef;
    using Codec = Lucene.Net.Codecs.Codec;
    using Directory = Lucene.Net.Store.Directory;
    using Document = Documents.Document;
    using LineFileDocs = Lucene.Net.Util.LineFileDocs;
    using LuceneTestCase = Lucene.Net.Util.LuceneTestCase;
    using MockAnalyzer = Lucene.Net.Analysis.MockAnalyzer;
    using NumericDocValuesField = NumericDocValuesField;
    using SortedSetDocValuesField = SortedSetDocValuesField;
    using TestUtil = Lucene.Net.Util.TestUtil;

    /// <summary>
    /// Compares one codec against another
    /// </summary>
    [TestFixture]
    public class TestDuelingCodecs : LuceneTestCase
    {
        private Directory leftDir;
        private IndexReader leftReader;
        private Codec leftCodec;

        private Directory rightDir;
        private IndexReader rightReader;
        private Codec rightCodec;

        private string info; // for debugging

        [SetUp]
        public override void SetUp()
        {
            base.SetUp();

            // for now its SimpleText vs Lucene46(random postings format)
            // as this gives the best overall coverage. when we have more
            // codecs we should probably pick 2 from Codec.availableCodecs()

            leftCodec = Codec.ForName("SimpleText");
            rightCodec = new RandomCodec(Random);

            leftDir = NewDirectory();
            rightDir = NewDirectory();

            long seed = Random.NextInt64();

            // must use same seed because of random payloads, etc
            int maxTermLength = TestUtil.NextInt32(Random, 1, IndexWriter.MAX_TERM_LENGTH);
            MockAnalyzer leftAnalyzer = new MockAnalyzer(new J2N.Randomizer(seed));
            leftAnalyzer.MaxTokenLength = maxTermLength;
            MockAnalyzer rightAnalyzer = new MockAnalyzer(new J2N.Randomizer(seed));
            rightAnalyzer.MaxTokenLength = maxTermLength;

            // but these can be different
            // TODO: this turns this into a really big test of Multi*, is that what we want?
            IndexWriterConfig leftConfig = NewIndexWriterConfig(TEST_VERSION_CURRENT, leftAnalyzer);
            leftConfig.SetCodec(leftCodec);
            // preserve docids
            leftConfig.SetMergePolicy(NewLogMergePolicy());

            IndexWriterConfig rightConfig = NewIndexWriterConfig(TEST_VERSION_CURRENT, rightAnalyzer);
            rightConfig.SetCodec(rightCodec);
            // preserve docids
            rightConfig.SetMergePolicy(NewLogMergePolicy());

            // must use same seed because of random docvalues fields, etc
            RandomIndexWriter leftWriter = new RandomIndexWriter(new J2N.Randomizer(seed), leftDir, leftConfig);
            RandomIndexWriter rightWriter = new RandomIndexWriter(new J2N.Randomizer(seed), rightDir, rightConfig);

            int numdocs = AtLeast(100);
            CreateRandomIndex(numdocs, leftWriter, seed);
            CreateRandomIndex(numdocs, rightWriter, seed);

            leftReader = MaybeWrapReader(leftWriter.GetReader());
            leftWriter.Dispose();
            rightReader = MaybeWrapReader(rightWriter.GetReader());
            rightWriter.Dispose();

            // check that our readers are valid
            TestUtil.CheckReader(leftReader);
            TestUtil.CheckReader(rightReader);

            info = "left: " + leftCodec.ToString() + " / right: " + rightCodec.ToString();
        }

        [TearDown]
        public override void TearDown()
        {
            if (leftReader != null)
            {
                leftReader.Dispose();
            }
            if (rightReader != null)
            {
                rightReader.Dispose();
            }

            if (leftDir != null)
            {
                leftDir.Dispose();
            }
            if (rightDir != null)
            {
                rightDir.Dispose();
            }

            base.TearDown();
        }

        /// <summary>
        /// populates a writer with random stuff. this must be fully reproducable with the seed!
        /// </summary>
        public static void CreateRandomIndex(int numdocs, RandomIndexWriter writer, long seed)
        {
            Random random = new J2N.Randomizer(seed);
            // primary source for our data is from linefiledocs, its realistic.
            LineFileDocs lineFileDocs = new LineFileDocs(random);

            // LUCENENET: compile a regex so we don't have to do it in each loop (for regex.split())
            Regex whiteSpace = new Regex("\\s+", RegexOptions.Compiled);

            // TODO: we should add other fields that use things like docs&freqs but omit positions,
            // because linefiledocs doesn't cover all the possibilities.
            for (int i = 0; i < numdocs; i++)
            {
                Document document = lineFileDocs.NextDoc();
                // grab the title and add some SortedSet instances for fun
                string title = document.Get("titleTokenized");
                string[] split = whiteSpace.Split(title).TrimEnd();
                foreach (string trash in split)
                {
                    document.Add(new SortedSetDocValuesField("sortedset", new BytesRef(trash)));
                }
                // add a numeric dv field sometimes
                document.RemoveFields("sparsenumeric");
                if (random.Next(4) == 2)
                {
                    document.Add(new NumericDocValuesField("sparsenumeric", random.NextInt64()));
                }
                writer.AddDocument(document);
            }

            lineFileDocs.Dispose();
        }

        /// <summary>
        /// checks the two indexes are equivalent
        /// </summary>
        [Test]
        public virtual void TestEquals()
        {
            AssertReaderEquals(info, leftReader, rightReader);
        }

        [Test]
        [Nightly]
        [LuceneNetSpecific]
        public void TestEquals_GH_545()
        {
            Codec leftCodec = Codec.ForName("SimpleText");
            Codec rightCodec = Codec.ForName("Lucene46");

            Directory leftDir = new Store.RAMDirectory();
            Directory rightDir = new Store.RAMDirectory();

            int maxTermLength = 21678;

            Analysis.Analyzer leftAnalyzer = new Analysis.Standard.StandardAnalyzer(TEST_VERSION_CURRENT) { MaxTokenLength = maxTermLength };
            Analysis.Analyzer rightAnalyzer = new Analysis.Standard.StandardAnalyzer(TEST_VERSION_CURRENT) { MaxTokenLength = maxTermLength };

            IndexWriterConfig leftConfig = new IndexWriterConfig(TEST_VERSION_CURRENT, leftAnalyzer) { Codec = leftCodec };
            IndexWriterConfig rightConfig = new IndexWriterConfig(TEST_VERSION_CURRENT, rightAnalyzer) { Codec = rightCodec };

            using IndexWriter leftWriter = new IndexWriter(leftDir, leftConfig);
            using IndexWriter rightWriter = new IndexWriter(rightDir, rightConfig);

            // Write indexes
            FieldType ft = new FieldType(TextField.TYPE_STORED)
            {
                StoreTermVectors = true,
                StoreTermVectorOffsets = true,
                StoreTermVectorPositions = true,
            };

            string body = "Esityslistalla on seuraavana Markovin laatima talous- ja raha-asioiden valiokunnan mietint\u00f6 (A5-0421/2002) Euroopan j\u00e4lleenrakennus- ja kehityspankin (EBRD) toiminnasta (2002/2095(INI)). PARSEP . (FR) Arvoisa puhemies, haluaisin ensimm\u00e4iseksi kiitt\u00e4\u00e4 Euroopan parlamenttia, talous- ja raha-asioiden valiokuntaa sek\u00e4 varsinkin esittelij\u00e4 Markovia mielenkiinnosta EBRD:\u00e4\u00e4 kohtaan. Kuten totesitte, t\u00e4m\u00e4 on ensimm\u00e4inen ja hyvin t\u00e4rke\u00e4 tapaaminen t\u00e4lle toimielimelle, jota minulla on kunnia johtaa. PARSEP Teen nyt vain nelj\u00e4 lyhytt\u00e4 ja selke\u00e4\u00e4 huomautusta, sill\u00e4 esittelij\u00e4 Markovin mietint\u00f6 ja komission j\u00e4senen Solbes Miran esitys ovat jo valaisseet asiaa parlamentille. PARSEP Ensimm\u00e4inen huomautukseni koskee itse EBRD:\u00e4\u00e4. EBRD on uudentyyppinen toimielin jo siksi, ett\u00e4 se on hiljattain perustettu ja siten melko uusi. Se on onnistunut mutta my\u00f6s ep\u00e4onnistunut teht\u00e4viss\u00e4\u00e4n, ja se haluaa oppia jatkuvasti uutta. Euroopan valtioilla ja Euroopan unionilla on t\u00e4ss\u00e4 toimielimess\u00e4 suuri enemmist\u00f6 yhdess\u00e4 pankin kumppanuusmaiden ja monien Euroopan ulkopuolisten maiden kanssa. N\u00e4it\u00e4 ovat muun muassa Yhdysvallat, Kanada, Japani, Australia, Uusi-Seelanti, Meksiko, Egypti, Israel ja Turkki. Kyse on toimielimest\u00e4, jossa Euroopalla on aivan olennainen asema monien kumppanien kanssa. Mik\u00e4 on t\u00e4m\u00e4n yhteisty\u00f6n perusta? Minusta niit\u00e4 on kaksi. PARSEP Ensinn\u00e4kin EBRD on vahvasti sitoutunut takaamaan it\u00e4\u00e4 koskevan siirtym\u00e4prosessin. T\u00e4h\u00e4n pyrkiv\u00e4t kaikki, mik\u00e4 tuli ilmi siin\u00e4 yhteydess\u00e4, kun strategiaamme p\u00e4ivitettiin tulevaisuutta varten ja kun Euroopan ulkopuoliset maat sitoutuivat tukemaan vastaisuudessa t\u00e4t\u00e4 siirtym\u00e4prosessia koko alueella, my\u00f6s Keski-Euroopassa. Toisen perustan muodostavat esittelij\u00e4 Markovin mainitsemat periaatteet: yhteisrahoitus yksityissektorin kanssa sek\u00e4 riskinottovalmiudet, jotta yksityinen sektori saataisiin kehitt\u00e4m\u00e4\u00e4n markkinataloutta siten, ettei yksityinen sektori kilpailisi julkisen sektorin kanssa. N\u00e4m\u00e4 t\u00e4ydent\u00e4vyys- ja ehdollisuusperiaatteet sek\u00e4 siirtym\u00e4prosessiin vaikuttamista koskevat periaatteet muodostavat EBRD:n todellisen pohjan. Minun on mainittava viel\u00e4 kolmas ulottuvuus, jonka esittelij\u00e4 mainitsi hetki sitten, nimitt\u00e4in poliittinen ulottuvuus. EBRD:ll\u00e4 on poliittinen teht\u00e4v\u00e4. Teht\u00e4v\u00e4n\u00e4mme on varmistaa paitsi markkinatalouden my\u00f6s demokratian kehittyminen, mink\u00e4 takia teemme pankin hallintoneuvoston - jonka p\u00e4\u00e4telm\u00e4t ovat julkisia - sis\u00e4isten neuvottelujen perusteella p\u00e4\u00e4t\u00f6ksi\u00e4 esimerkiksi siit\u00e4, ett\u00e4 rahoitusta v\u00e4hennet\u00e4\u00e4n tai se jopa lopetetaan Valko-Ven\u00e4j\u00e4n ja Turkmenistanin kaltaisten maiden kohdalla. PARSEP Toinen huomioni koskee hankittua kokemusta ja strategiaa, joista esittelij\u00e4 Markov puhui. Pankin toiminta kasvaa jatkuvasti. Viime vuonna investoimme 3,9 miljardia euroa, mik\u00e4 on enemm\u00e4n kuin koskaan. Noin kymmeness\u00e4 vuodessa olemme saaneet houkuteltua investointeja it\u00e4\u00e4n 70-80 miljardin euron arvosta, mik\u00e4 on huomattava osa tuolle alueelle suuntautuvasta investointivirrasta. Toteutamme parhaillaan my\u00f6s EBRD:n osakkaiden hyv\u00e4ksym\u00e4\u00e4, It\u00e4-Euroopan maihin keskittyv\u00e4\u00e4 strategiaa. PARSEP Olemme vahvasti sitoutuneet Keski-Euroopan maihin. Syyt voimakkaalle investointitarpeelle ovat selv\u00e4t - teh\u00e4n tunnette Keski-Euroopan tilanteen, josta minulla oli tilaisuus puhua pitk\u00e4\u00e4n komission j\u00e4senen Solbes Miran kanssa - joten varsinkin budjettipaine voi hellitt\u00e4\u00e4 vain huomattavilla investoinneilla. T\u00e4m\u00e4 on luonnollisesti hyvin t\u00e4rke\u00e4\u00e4 EBRD:lle, joka keskittyy ensisijaisiin teht\u00e4viin, joita ovat pk-yritysten rahoitus, keskisuurten yritysten rahoitus, teollisuuden rakenneuudistus sek\u00e4 alueellinen kehitys. Pid\u00e4mme toki itse\u00e4mme l\u00e4hinn\u00e4 elimen\u00e4, joka panee t\u00e4yt\u00e4nt\u00f6\u00f6n hankkeita komission ja Euroopan unionin puolesta. Kehit\u00e4mme nyt toimintaamme my\u00f6s Kaakkois-Euroopassa - Balkanilla, Kaukasuksella, Ukrainassa, Romaniassa ja Bulgariassa - sek\u00e4 erityisesti Ven\u00e4j\u00e4ll\u00e4. PARSEP Toinen strategiaa koskeva huomioni koskee pienten yritysten, mikroyritysten ja pk-yritysten kehitt\u00e4miseen kiinnitt\u00e4m\u00e4\u00e4mme huomiota. PARSEP T\u00e4m\u00e4 oli viel\u00e4 kymmenisen vuotta sitten tuntematon k\u00e4site It\u00e4-Euroopassa, mutta nyt se on jo yleistynyt. Toteutamme sit\u00e4 etenkin - haluan korostaa t\u00e4t\u00e4 - osakkaidemme tuen ja Euroopan unionin ansiosta. Niill\u00e4 on t\u00e4ss\u00e4 aivan olennainen asema. Edist\u00e4mme n\u00e4iden toimien toteuttamista ja haluamme ty\u00f6skennell\u00e4 yhdess\u00e4 pankkisektorin kanssa sen asiantuntemuksen parantamiseksi siten, ett\u00e4 pankkihenkil\u00f6kuntaa koulutetaan kehitt\u00e4m\u00e4\u00e4n n\u00e4it\u00e4 tekniikoita. Jos t\u00e4m\u00e4 ei onnistu, perustamme omia elimi\u00e4 kehityksen varmistamiseksi vaaditussa ajassa. PARSEP Kehit\u00e4mme samaan tapaan suhdetoimintaamme kaikkien L\u00e4nsi-Euroopan ja Japanin yritysten kanssa. Miksi kehit\u00e4mme sit\u00e4? Kehit\u00e4mme sit\u00e4 ensinn\u00e4kin siksi, ett\u00e4 kyse on veronmaksajien rahoista. Lis\u00e4ksi kyseiset yritykset ovat investoineet teid\u00e4n kauttanne EBRD:\u00e4\u00e4n, jolloin on t\u00e4ysin luonnollista, ett\u00e4 edes kerromme n\u00e4ille yrityksille, mit\u00e4 voimme pankin avulla tehd\u00e4. Kehit\u00e4mme yhteyksi\u00e4 paitsi suuriin my\u00f6s keskisuuriin yrityksiin Euroopassa, sill\u00e4 meist\u00e4 keskisuurilla yrityksill\u00e4 voi olla \u00e4\u00e4rimm\u00e4isen t\u00e4rke\u00e4 rooli It\u00e4-Euroopan talouksien kehitt\u00e4jin\u00e4. PARSEP Samaan tapaan ty\u00f6skennelless\u00e4mme suurten yritysten kanssa seuraamme, miten pienten yritysten kehitt\u00e4minen suuryritysten avulla vaikuttaisi rakentavasti It\u00e4-Euroopan talouksiin ja etenkin alihankintatoimintaan. Paras esimerkki t\u00e4st\u00e4 on mielest\u00e4ni yritysten kehitt\u00e4minen autoteollisuudessa, joka houkuttelee alihankkijoita ja mahdollistaa paikallisen alihankintatoiminnan kehitt\u00e4misen. PARSEP Viimeisen\u00e4 huomiona sanoisin, ett\u00e4 ponnistelemme parhaillaan lujasti kehitt\u00e4\u00e4ksemme yhteisty\u00f6t\u00e4 paikallisten yritysten kanssa, sill\u00e4 t\u00e4m\u00e4 on meist\u00e4 \u00e4\u00e4rimm\u00e4isen t\u00e4rke\u00e4\u00e4. T\u00e4ss\u00e4 on n\u00e4ht\u00e4viss\u00e4 kehityst\u00e4 kaikkialla Keski-Euroopassa, mutta my\u00f6s Ven\u00e4j\u00e4ll\u00e4 ja Keski-Aasiassa. T\u00e4ten ryhdymme kehitt\u00e4m\u00e4\u00e4n kuntien rahoitusta, ja aiomme tehd\u00e4 sen paikallisessa valuutassa, vaikka otamme rahoitusmarkkinoiden kehityksen takia itse valuuttariskin paikallisten yhteis\u00f6jen puolesta. PARSEP Nelj\u00e4s huomioni koskee esittelij\u00e4nne tekemi\u00e4 ehdotuksia ja huomautuksia, joista haluan vain sanoa olevani samaa mielt\u00e4. Hyv\u00e4ksymme ne, otamme ne huomioon ja pyrimme toteuttamaan niit\u00e4 my\u00f6s k\u00e4yt\u00e4nn\u00f6ss\u00e4. Teen kuitenkin kaksi erityishuomiota. Ensinn\u00e4kin sosiaalisesta ulottuvuudesta, joka otetaan aina huomioon EBRD:ss\u00e4, joka pit\u00e4\u00e4 sosiaalista yhteenkuuluvuutta t\u00e4rke\u00e4n\u00e4 asiana, vaikka se on markkinatalouden edist\u00e4miseen tarkoitettu toimielin. Ty\u00f6llisyysasiat ja erityisesti hinnoittelupolitiikka ovat meille hyvin t\u00e4rkeit\u00e4 kysymyksi\u00e4. Kansalaisten elinolojen parantaminen, juomakelpoinen vesi, liikenne sek\u00e4 kunnallispalvelut kokonaisuudessaan ovat osa hinnoittelupolitiikkaa. Teh\u00e4n tied\u00e4tte, ettei It\u00e4-Euroopan mailla ole kokemusta t\u00e4llaisesta politiikasta. N\u00e4in ollen terveen ja kest\u00e4v\u00e4n sek\u00e4 kansalaisten hyv\u00e4ksym\u00e4n hinnoittelupolitiikan harjoittaminen on elinehto. T\u00e4lt\u00e4 osin haluankin kiitt\u00e4\u00e4 Euroopan unionia rahoituksesta, jota se on my\u00f6nt\u00e4nyt meille l\u00e4hinn\u00e4 yhteis\u00f6n rahastoista, joiden ansiosta on voitu alentaa hankkeiden alkukustannuksia ja siten toteuttaa kansalaisten silmiss\u00e4 hyv\u00e4ksytt\u00e4v\u00e4\u00e4 hinnoittelupolitiikkaa ja s\u00e4ilytt\u00e4\u00e4 palvelujen ja investointien taso. PARSEP Toinen huomioni koskee ymp\u00e4rist\u00f6kysymyksi\u00e4. Ymp\u00e4rist\u00f6 on keskeisell\u00e4 sijalla EBRD:n teht\u00e4vien joukossa. Ty\u00f6kentt\u00e4 on laaja. Menneisyyden painolasti on suuri It\u00e4-Euroopan maissa, mutta mainitsen vain toimintamme yhden olennaisen puolen, nimitt\u00e4in ydinvoimaloiden purkamisen, jonka parissa teemme paljon ty\u00f6t\u00e4 Euroopan unionin kanssa. Hoidamme 1,5 miljardin euron suuruisia rahastoja, jotka on tarkoitettu ydinvoimaloiden puhdistushankkeisiin sek\u00e4 Euroopan unioniin pian liittyviss\u00e4 maissa ett\u00e4 my\u00f6hemmin unioniin liittyviss\u00e4 maissa. T\u00e4m\u00e4 on meille olennainen toiminnan ala. PARSEP Viimeinen huomioni koskee suhteitamme Euroopan unioniin. Euroopan unioni on merkitt\u00e4v\u00e4 yhteisty\u00f6kumppanimme ja tukijamme etenkin teid\u00e4n panoksenne ansiosta, arvoisa komission j\u00e4sen Solbes. H\u00e4n on pankkimme hallintoneuvoston puheenjohtaja, joka valvoo ty\u00f6t\u00e4mme tiiviisti ja vaaditulla tiukkuudella. Ty\u00f6skentelemme yhdess\u00e4 Euroopan unionin toimielinten yksik\u00f6iden ja kaikkien j\u00e4senvaltioiden yksik\u00f6iden kanssa avoimesti ja ihanteellisissa oloissa. Kuten totesittekin, komission j\u00e4sen Solbes, saamme teilt\u00e4 meille hyvin olennaista poliittista tukea mutta my\u00f6s taloudellista tukea, ja teid\u00e4n teht\u00e4v\u00e4\u00e4nne laajennetaan parhaillaan. PARSEP Toinen yhteisty\u00f6kumppanimme on Euroopan investointipankki EIP, joka on meille t\u00e4rke\u00e4 kumppani. Olemme tyytyv\u00e4isi\u00e4 siihen, ett\u00e4 se on lis\u00e4nnyt ehdokasvaltioille tarkoitettua v\u00e4ltt\u00e4m\u00e4t\u00f6nt\u00e4 rahoitusta. Ty\u00f6skentelen paljon yhdess\u00e4 Philippe Maystadtin kanssa, ja meill\u00e4 on tiiviit suhteet. Teh\u00e4n tied\u00e4tte, ett\u00e4 EIP on komission ja Euroopan unionin tavoin EBRD:n osakas. N\u00e4in ollen suhteet ovat tiiviit ja luottamukselliset. Tavoitteemme ei todellakaan ole luoda p\u00e4\u00e4llekk\u00e4isi\u00e4 t\u00f6it\u00e4, vaan varmistaa v\u00e4ltt\u00e4m\u00e4t\u00f6n t\u00f6iden t\u00e4ydent\u00e4vyys. Vahvistaisin viel\u00e4 sen, ett\u00e4 niiden suhteiden taso, jotka minulla on Philippe Maystadtiin ja EIP:hen, on paras tae Euroopan veronmaksajille siit\u00e4, ett\u00e4 n\u00e4iden molempien toimielinten k\u00e4ytt\u00f6\u00f6n annetut varat k\u00e4ytet\u00e4\u00e4n asianmukaisella tavalla. PARSEP Pahoittelen hieman liian pitk\u00e4\u00e4 puheenvuoroani. Minusta oli t\u00e4rke\u00e4\u00e4 tehd\u00e4 n\u00e4m\u00e4 muutamat alustavat huomiot t\u00e4\u00e4ll\u00e4 Euroopan parlamentissa, jossa meille tarjoutui ensimm\u00e4ist\u00e4 kertaa mahdollisuus k\u00e4sitell\u00e4 EBRD:n asemaa. PARSEP Arvoisa puhemies, min\u00e4kin haluan toivottaa EBRD:n p\u00e4\u00e4johtaja Lemierren ja h\u00e4nen kollegansa tervetulleiksi parlamenttiin. H\u00e4nen puheenvuorostaan sai sellaisen vaikutelman, ett\u00e4 h\u00e4n ja h\u00e4nen kollegansa suhtautuvat ty\u00f6h\u00f6ns\u00e4 osittain idealistisesti, mutta my\u00f6s hyvin k\u00e4yt\u00e4nn\u00f6nl\u00e4heisesti. PARSEP Haluaisin onnitella my\u00f6s esittelij\u00e4 Markovia h\u00e4nen mietinn\u00f6st\u00e4\u00e4n. H\u00e4n on omistanut paljon huolta, tunnollista ty\u00f6t\u00e4 ja j\u00e4rkevi\u00e4 ajatuksia Euroopan j\u00e4lleenrakennus- ja kehityspankin roolin ja toimintakyvyn arviointiin. H\u00e4nen kommenttinsa ovat harkittuja, ja niill\u00e4 on my\u00f6nteinen vaikutus EBRD:n tulevaisuuteen. PARSEP N\u00e4in vastik\u00e4\u00e4n er\u00e4\u00e4ss\u00e4 lehdess\u00e4 otsikon, jossa todettiin, ett\u00e4 EBRD tunnetaan huonosti. Koska pankki on viel\u00e4 melko uusi, se ei varmaan pahastu t\u00e4st\u00e4 ilmauksesta, mutta jos t\u00e4m\u00e4 tosiaan pit\u00e4\u00e4 paikkansa, on korkea aika lis\u00e4t\u00e4 huomattavasti EBRD:n tunnettuutta. Ehk\u00e4p\u00e4 sen esittely parlamentissa auttaa t\u00e4ss\u00e4. Kuten t\u00e4\u00e4ll\u00e4 on jo todettu, parlamentissa on nyt ensimm\u00e4ist\u00e4 kertaa esitelty EBRD:n toimintaa, vaikka enemmist\u00f6 pankin p\u00e4\u00e4omasta ja \u00e4\u00e4nist\u00e4 kuuluu EU:n j\u00e4senvaltioille ja EU:n toimielimille ja vaikka EBRD:ll\u00e4 on ollut ja on vastedeskin t\u00e4rke\u00e4 rooli siin\u00e4, ett\u00e4 se auttaa monia ehdokasvaltioita sopeuttamaan niiden talouksia ja poliittisia elimi\u00e4 vaatimuksiin, joita n\u00e4iden ehdokasvaltioiden kansalaiset tavoittelevat ja jotka mahdollistavat n\u00e4iden valtioiden kitkattoman liittymisen Euroopan unioniin. PARSEP T\u00e4ss\u00e4 ei ollut kaikki. EBRD tuo my\u00f6s ratkaisevan panoksen laajentuneen Euroopan unionin rajavaltioille. On hyvin mielenkiintoista, ett\u00e4 EBRD julkisti viime marraskuussa Ven\u00e4j\u00e4n-operaatioitaan koskevan uuden strategian, jonka mukaan sen toiminta laajenee my\u00f6s muille Ven\u00e4j\u00e4n alueille kuin Moskovaan ja Pietariin. T\u00e4m\u00e4 monipuolistaisi Ven\u00e4j\u00e4n taloutta, joka ei siten en\u00e4\u00e4 olisi liian riippuvainen luonnonvaroista. EBRD on suurin yksitt\u00e4inen ulkomainen investoija Ven\u00e4j\u00e4ll\u00e4, ja Ven\u00e4j\u00e4 on suurin maa sen teht\u00e4v\u00e4kent\u00e4ss\u00e4. PARSEP Kiinnit\u00e4mme Euroopan unionissa huomiota my\u00f6s EBRD:n toimintaan Jugoslaviassa, Jugoslavian entisiss\u00e4 tasavalloissa sek\u00e4 Balkanilla. Mantereemme rauhan ja turvallisuuden kannalta on ratkaisevan t\u00e4rke\u00e4\u00e4 auttaa n\u00e4it\u00e4 maita saavuttamaan sellainen taloudellisen ja poliittisen vakauden tila, jonka pohjalta my\u00f6s ne voivat jouduttaa Euroopan unioniin liittymist\u00e4 aikanaan. PARSEP EBRD:n tavoitteet ovat ainutlaatuisia t\u00e4llaiseksi toimielimeksi siin\u00e4 mieless\u00e4, ett\u00e4 se pyrkii saamaan aikaan taloudellista kehityst\u00e4, mutta sill\u00e4 on my\u00f6s erityinen velvollisuus auttaa asiakasmaitaan kehitt\u00e4m\u00e4\u00e4n demokraattista kulttuuria ja v\u00e4ltt\u00e4m\u00e4tt\u00f6mi\u00e4 demokraattisia hallintoelimi\u00e4 ja menettelyj\u00e4. Sen toiminnan onnistumisen valvonnassa ja arvioinnissa on siten otettava huomioon, miten hyvin se on saavuttanut taloudelliset ja demokraattiset tavoitteensa. En voi kuin h\u00e4mm\u00e4stell\u00e4 EBRD:n saavutuksista. PARSEP PPE-DE-ryhm\u00e4 aikoo tukea Markovin mietint\u00f6\u00e4 sellaisenaan. Aiomme \u00e4\u00e4nest\u00e4\u00e4 kahta lukuun ottamatta kaikkia k\u00e4sitelt\u00e4viksi j\u00e4tettyj\u00e4 tarkistuksia vastaan, sill\u00e4 ne eiv\u00e4t paranna mietint\u00f6\u00e4 mitenk\u00e4\u00e4n. Poikkeuksena mainittakoon hiljattain k\u00e4ynnistetty julkinen tiedotuspolitiikka, jota kannatamme. N\u00e4in ollen annamme tukemme tarkistuksille 2 ja 4, joilla kannatetaan ja edistet\u00e4\u00e4n entist\u00e4 avoimempaa l\u00e4hestymistapaa. PARSEP Kun parlamentti seuraavan kerran laatii mietinn\u00f6n EBRD:st\u00e4, se voi toivoakseni kiitt\u00e4\u00e4 siit\u00e4 jatkuvasta menestyksest\u00e4, jota EBRD:n tunnetuksi tekemisess\u00e4 ja laajan arvostuksen saamisessa on saatu aikaan. Toivon EBRD:lle, sen p\u00e4\u00e4johtajalle Lemierrelle sek\u00e4 sen henkil\u00f6kunnalle ja asiakkaille jatkuvaa menestyst\u00e4 vuonna 2003 ja my\u00f6hemmin tulevaisuudessa. Menestys on meille kaikille olennaisen t\u00e4rke\u00e4\u00e4. PARSEP Arvoisa puhemies, arvoisa komission j\u00e4sen, arvoisat parlamentin j\u00e4senet, arvoisa EBRD:n p\u00e4\u00e4johtaja, olen iloinen siit\u00e4, ett\u00e4 parlamentissa k\u00e4sitell\u00e4\u00e4n nyt ensimm\u00e4ist\u00e4 kertaa teid\u00e4n l\u00e4sn\u00e4 ollessanne mietint\u00f6\u00e4 EBRD:st\u00e4, jonka toiminta, maine ja uskottavuus ovat selv\u00e4sti vahvistuneet p\u00e4\u00e4johtajakaudellanne. Haluan kiitt\u00e4\u00e4 my\u00f6s esittelij\u00e4\u00e4mme, sill\u00e4 h\u00e4n on osoittanut, miten perinpohjaisesti parlamentti on tarkastellut EBRD:n toimintaa. Olemme yhdess\u00e4 esittelij\u00e4n kanssa allekirjoittaneet kaksi tarkistusta, jotka parlamentti voisi toivoakseni hyv\u00e4ksy\u00e4 talous- ja raha-asioiden valiokunnan ty\u00f6n parantamiseksi entisest\u00e4\u00e4n. PARSEP EBRD:n toiminta ansaitsee parlamentin kiitoksen, sill\u00e4 EBRD on ainutlaatuinen ja esimerkillinen toimielin. Ainutlaatuiseksi sen tekev\u00e4t sit\u00e4 eteenp\u00e4in ajavat periaatteet, joita ovat siirtym\u00e4prosessin tukeminen, t\u00e4ydent\u00e4vyysperiaate ja moitteeton pankkitoiminta. T\u00e4llaisesta esimerkillisyydest\u00e4 voisimme mielest\u00e4ni ottaa opiksi monilla muilla aloilla. EBRD:ll\u00e4 on my\u00f6s kyky yhdist\u00e4\u00e4 toisiinsa taloudelliset ja poliittiset kriteerit, mik\u00e4 on toinen t\u00e4rke\u00e4 tekij\u00e4. Kyse ei ole sen enemp\u00e4\u00e4 eurooppalaisesta pankista kuin Maailmanpankin sivuhaarasta, mutta silti suhteenne meihin ovat tiiviit. EBRD kykenee my\u00f6s ottamaan riskej\u00e4, kehittym\u00e4\u00e4n ep\u00e4vakaassa toimintaymp\u00e4rist\u00f6ss\u00e4 ja varmistamaan v\u00e4ltt\u00e4m\u00e4tt\u00f6mien hankkeiden rahoituksen my\u00f6s kriisitilanteissa, kuten osoititte Ven\u00e4j\u00e4n kriisin aikana. Kaiken lis\u00e4ksi EBRD kykenee jatkuvasti edist\u00e4m\u00e4\u00e4n pk-yritysten toimintaa, johon parlamentti kiinnitt\u00e4\u00e4 erityist\u00e4 huomiota. PARSEP Haluaisin korostaa erityisesti kahta kohtaa Markovin mietinn\u00f6ss\u00e4. Ensimm\u00e4inen liittyy siihen, ett\u00e4 vaikka EBRD ei ole Euroopan unionin toimielin mutta koska sill\u00e4 on unioniin erityiset suhteet, meist\u00e4 on t\u00e4rke\u00e4\u00e4 voida hy\u00f6dynt\u00e4\u00e4 t\u00e4t\u00e4 v\u00e4linett\u00e4, jotta unionin ulkopuolella voidaan edist\u00e4\u00e4 unionin periaatteita ja sit\u00e4 strategiaa, jonka Euroopan unioni laati sek\u00e4 Lissabonissa ett\u00e4 G\u00f6teborgissa etenkin ty\u00f6llisyyden, sosiaalipolitiikan, ymp\u00e4rist\u00f6kysymysten, innovaatioiden ja tietopohjaisen talouden kehitt\u00e4miseksi kest\u00e4v\u00e4n kehityksen hyv\u00e4ksi. Toinen huomioni koskee tarkistuksia, joille toivon parlamentin tukea. Niill\u00e4 pyrit\u00e4\u00e4n lis\u00e4\u00e4m\u00e4\u00e4n entisest\u00e4\u00e4n ty\u00f6llisyyskriteerin merkityst\u00e4 ja ammattiliittojen huomioon ottamista yhteisrahoituksen piiriin kuuluvien hankkeiden arvioinnissa, vaikka tied\u00e4n, ett\u00e4 n\u00e4in tehd\u00e4\u00e4n jo. EBRD:n ei pid\u00e4 edist\u00e4\u00e4 talousliberalismia, vaan sen olisi edistett\u00e4v\u00e4 my\u00f6s demokraattisia ja sosiaalisia arvoja, sill\u00e4 sit\u00e4 ei ole perustettu IMF:n mallin mukaisesti. PARSEP Korostaisin, ett\u00e4 EBRD:n toiminnalla on huomattava merkitys korruption torjunnassa ja hyvien k\u00e4yt\u00e4nteiden edist\u00e4misess\u00e4 ja sen korostamisessa, ett\u00e4 on v\u00e4ltett\u00e4v\u00e4 sellaisten j\u00e4rjestelmien ja toimijoiden rahoittamista, joiden rahoitusl\u00e4hteit\u00e4 on syyt\u00e4 ep\u00e4ill\u00e4. Kuten mainitsitte, t\u00e4m\u00e4 on saanut teid\u00e4t lopettamaan toimintanne Valko-Ven\u00e4j\u00e4ll\u00e4. Korruptiota esiintyy my\u00f6s muualla, kuten tied\u00e4tte. PARSEP Lopuksi korostaisin EBRD:n tulevaisuutta, sill\u00e4 vaikka kymmenen niist\u00e4 maista, joissa teill\u00e4 on toimintaa, liittyy piakkoin unioniin 1. toukokuuta 2004 lukien, siirtym\u00e4kausi on viel\u00e4 pahasti kesken. N\u00e4in ollen tarvitaan viel\u00e4 paljon EBRD:n tukea siirtym\u00e4kauden rahoituksen helpottamiseksi n\u00e4iss\u00e4 maissa, jotta niist\u00e4 tulisi Euroopan unionin t\u00e4ysivaltaisia j\u00e4seni\u00e4. PARSEP Arvoisa puhemies, arvoisa komission j\u00e4sen, arvoisa EBRD:n p\u00e4\u00e4johtaja Lemierre, haluaisin aloittaa kiitt\u00e4m\u00e4ll\u00e4 teit\u00e4 sek\u00e4 henkil\u00f6kuntaanne ty\u00f6st\u00e4nne. Tase on vaikuttava. Olemme tyytyv\u00e4isi\u00e4 siihen, ettei t\u00e4n\u00e4\u00e4n esitt\u00e4m\u00e4nne tase anna aihetta kritiikkiin. PARSEP Toiseksi kannatan ensisijaisia tavoitteitanne. Erityisest\u00e4 teht\u00e4v\u00e4st\u00e4nne huolimatta minusta on hyvin t\u00e4rke\u00e4\u00e4, ett\u00e4 - kuten olette todennut - EBRD vastaa my\u00f6s komission ensisijaisten tavoitteiden t\u00e4yt\u00e4nt\u00f6\u00f6npanosta sek\u00e4 komission ja parlamentin v\u00e4lisest\u00e4 yhteisty\u00f6st\u00e4. Meid\u00e4n ei pid\u00e4 levitt\u00e4yty\u00e4 joka puolelle liian heikoin perustein. Meid\u00e4n on asetettava ensisijaisia tavoitteita poliittisten ja taloudellisten velvollisuuksiemme t\u00e4ytt\u00e4miseksi maailmassa - j\u00e4senvaltioissa, ehdokasvaltioissa ja l\u00e4hiseudulla. Emme voi asettaa toisistaan poikkeavia ensisijaisia tavoitteita. PARSEP Minusta seuraavat kysymykset ovat t\u00e4rkeit\u00e4. Toivon, ett\u00e4 kerrotte seuraavalla kerralla my\u00f6s toimintanne vaikutuksista siirtym\u00e4vaiheessa oleviin maihin, sill\u00e4 tiet\u00e4myst\u00e4 pankin vahvuuksista lis\u00e4isi, jos saisimme taseen lis\u00e4ksi tietoa my\u00f6s pankin toiminnan makrotaloudellisista vaikutuksista. PARSEP Toiseksi minusta on t\u00e4rke\u00e4\u00e4, ett\u00e4 jatkaisitte ja tehostaisitte toimintaanne pienten ja keskisuurten yritysten tukemiseksi. Olemme kaikki tietoisia pk-yritysten erityisest\u00e4 roolista koulutusmahdollisuuksien ja ty\u00f6paikkojen luojana ja varmistajana. Oletamme teid\u00e4n tehostavan t\u00e4t\u00e4 sitoumusta etenkin keskittym\u00e4ll\u00e4 entist\u00e4 enemm\u00e4n pk-yrityksille tarkoitettuihin p\u00e4\u00e4omaosakkuuksiin ja tukemalla pk-yritysten infrastruktuurin kehitt\u00e4mist\u00e4. Huomauttaisin viel\u00e4, ett\u00e4 Ven\u00e4j\u00e4ll\u00e4 ainoastaan 10 prosenttia ty\u00f6voimasta sijoittuu pk-yrityksiin. PARSEP Olemme viikon kuluessa puhuneet paljon Lissabonin prosessista, joten on t\u00e4rke\u00e4\u00e4, ett\u00e4 ottaisitte sen huomioon arvioidessanne hankkeita ja yrityksi\u00e4 ja ett\u00e4 ottaisitte sen yhdeksi kriteeriksenne. PARSEP Toki my\u00f6s ehdokasvaltiot on asetettava etusijalle. EU:n laajentumista ei ole viel\u00e4 kuitenkaan saatettu p\u00e4\u00e4t\u00f6kseen vuonna 2004. Balkanin kysymys - toisin sanoen entisen Jugoslavian kysymys - on t\u00e4rke\u00e4 Euroopan unionin kannalta, samoin kuin turvallisuusvy\u00f6hykkeen luominen. Talouksien kehitt\u00e4minen Euroopan unionin raja-alueilla on olennainen osa t\u00e4t\u00e4 prosessia. PARSEP N\u00e4in ollen kehottaisin teit\u00e4 jatkamaan toimintalinjaanne ja ottamaan parlamentin mielipiteet huomioon. Tuemme teit\u00e4 t\u00e4ss\u00e4 l\u00e4hestymistavassa. PARSEP Arvoisa puhemies, EBRD rahoitetaan veronmaksajien rahoilla, joten pankin toiminnan on oltava avointa. Olemme Euroopan parlamentissa pyrkineet avoimuuteen aina pankin perustamisesta asti, mutta minun on sanottava, ettei t\u00e4ss\u00e4 ole aina onnistuttu. Pankilla on ollut my\u00f6s hyvin vaikeita aikoja. My\u00f6s parlamentissa on esiintynyt eri\u00e4vi\u00e4 n\u00e4kemyksi\u00e4 siit\u00e4, tarvitsemmeko me yh\u00e4 todella omaa pankkia. T\u00e4m\u00e4kin olisi otettava huomioon t\u00e4m\u00e4np\u00e4iv\u00e4isess\u00e4 keskustelussa. PARSEP Arvoisa Lemierre, kuvailitte pankin teht\u00e4v\u00e4\u00e4 perinteiseksi yleishy\u00f6dyllisten palvelujen tarjoajaksi, joka hy\u00f6dynt\u00e4\u00e4 kuitenkin yksityisen sektorin interventiov\u00e4lineit\u00e4. Kuvaus on kielt\u00e4m\u00e4tt\u00e4 osuva, ja meid\u00e4n onkin arvioitava pankin toimintaa n\u00e4hd\u00e4ksemme, miten t\u00e4m\u00e4 n\u00e4kyy k\u00e4yt\u00e4nn\u00f6ss\u00e4. Olette ep\u00e4ilem\u00e4tt\u00e4 hankalassa asemassa, koska yksityisen sektorin kehitt\u00e4minen pankin kumppanuusmaissa tapahtuneiden onnistuneiden siirtym\u00e4prosessien tuloksena on ep\u00e4onnistunut l\u00e4hinn\u00e4 siksi, ett\u00e4 niist\u00e4 puuttuu t\u00e4ysivaltaisesti toimiva rahoitussektori, jolloin taloudelliset olot yritysten perustamisen ja toiminnan kannalta ovat kovin hankalat. T\u00e4m\u00e4n vuoksi minusta on hyv\u00e4, ett\u00e4 olette keskitt\u00e4neet 30 prosenttia resursseistanne t\u00e4lle sektorille. My\u00f6nn\u00e4n my\u00f6s, ett\u00e4 toimintanne pienten ja keskisuurten yritysten tukemiseksi on t\u00e4rke\u00e4\u00e4 toimivan markkinatalouden ja demokratian kehitt\u00e4misen kannalta. PARSEP On kuitenkin t\u00e4rke\u00e4\u00e4 ymm\u00e4rt\u00e4\u00e4 my\u00f6s se, kuinka n\u00e4ihin maihin saataisiin luotua vakaat olot. Miten talouskasvua voitaisiin edist\u00e4\u00e4 investoinnein? T\u00e4m\u00e4 kysymys askarruttaa meit\u00e4 jatkuvasti my\u00f6s Euroopan unionissa. Minusta on erityisen t\u00e4rke\u00e4\u00e4, ett\u00e4 voitte ottaa sellaisia riskej\u00e4, joihin muilla ei ole valmiuksia, ja ettei t\u00e4m\u00e4 riskinottovalmius ole ristiriidassa moitteettoman ja luotettavan yritysjohtamisen kanssa. T\u00e4st\u00e4 on osoituksena se, ett\u00e4 luokituslaitokset ovat my\u00f6nt\u00e4neet EBRD:lle korkean kelpoisuusluokituksen. EBRD on monenv\u00e4listen kehityspankkien joukossa ainutlaatuinen, koska sill\u00e4 on poliittinen teht\u00e4v\u00e4. Sen teht\u00e4v\u00e4n\u00e4 on edist\u00e4\u00e4 sosiaalista markkinataloutta, demokratiaa, moniarvoisuutta ja oikeusvaltiota. T\u00e4lt\u00e4 osin teid\u00e4n olisi syyt\u00e4 keskitty\u00e4 entist\u00e4kin voimakkaammin ihmisoikeuksien ja demokraattisten periaatteiden noudattamisen varmistamiseen. Teid\u00e4n olisi ev\u00e4tt\u00e4v\u00e4 aiempaa useammin tuki sellaisilta valtioilta ja yrityksilt\u00e4, jotka ovat mukana vilpillisess\u00e4 toiminnassa ja joissa esiintyy korruptiota ja huonoa hallintoa. Niin olisi toimittava my\u00f6s tapauksissa, joissa on ep\u00e4onnistuttu t\u00e4ytt\u00e4m\u00e4\u00e4n ymp\u00e4rist\u00f6n kannalta j\u00e4rkev\u00e4n, sosiaalisesti sopusointuisen ja kest\u00e4v\u00e4n kehityksen tavoitteet. PARSEP EBRD toimii edelleen Keski- ja It\u00e4-Euroopassa. N\u00e4iden valtioiden liittyminen EU:hun avaa monia sellaisia investointimahdollisuuksia, jotka eiv\u00e4t avaudu muille valtioille. Sen vuoksi kannattaa pohtia, olisiko EBRD:n, EIP:n ja Maailmanpankin v\u00e4list\u00e4 nykyist\u00e4 vastuunjakoa syyt\u00e4 tarkistaa my\u00f6s maantieteellisesti huolimatta niiden selkeist\u00e4 teht\u00e4vist\u00e4, sill\u00e4 entisten IVY-maiden siirtym\u00e4prosessin onnistuminen edist\u00e4isi n\u00e4iden maiden taloudellista ja poliittista vakautta, mutta vaikuttaisi my\u00f6s Euroopan unioniin ja muihin alueisiin maailmassa. Kaikkien maiden siirtym\u00e4prosesseissa edellytet\u00e4\u00e4n yhteisten toimien jatkamista ja rahoitustukea, ja t\u00e4m\u00e4 koskee my\u00f6s ehdokasvaltioita. Kaikkein v\u00e4hiten kehittyneet maat uhkaavat vakautta kuitenkin eniten. Arvoisa puhemies, t\u00e4ss\u00e4 on minusta uusi haaste EBRD:lle! PARSEP Arvoisa puhemies, Markovin erinomaisesta mietinn\u00f6st\u00e4 saa vaikutelman, ett\u00e4 EBRD on todella suuri menestys. Se on onnistunut esimerkiksi investoinneissaan ehdokasvaltioiden ja entisten neuvostotasavaltojen ydinturvallisuuteen. Kuten EBRD:n p\u00e4\u00e4johtaja totesi, pankki on tehnyt hienoa ty\u00f6t\u00e4 parantaessaan n\u00e4iden maiden ydinturvallisuutta. T\u00e4h\u00e4n on todellinen tarve. Totuus on, ett\u00e4 n\u00e4iden maiden kaikki ydinvoimalat ovat todellisia aikapommeja. Ajatellaanpa vaikka T?ernobylin onnettomuutta. N\u00e4iden ydinvoimaloiden turvallisuus on niin huono, ett\u00e4 on h\u00e4pe\u00e4, ettei niit\u00e4 ole suljettu jo aikoja sitten. EBRD vastaa Liettuassa sijaitsevan Ignalinan ydinvoimalan sulkemisen rahoituksesta. Olen k\u00e4ynyt tuossa ydinvoimalassa aika monta kertaa. Voimaloiden turvallisuus on niin kehno, ett\u00e4 niihin tekemieni vierailujen j\u00e4lkeen olen iloinen ajatuksesta, etten en\u00e4\u00e4 aio hankkia lapsia. PARSEP Liettua lupasi K\u00f6\u00f6penhaminan huippukokouksessa sulkea Ignalinan vuonna 2009. T\u00e4lt\u00e4 osin pankin on todella teht\u00e4v\u00e4 jotakin. Vastik\u00e4\u00e4n on esitetty ehdotus investointipaketista, jolla saataisiin selvyytt\u00e4 ydinj\u00e4tteen varastointimenetelm\u00e4\u00e4n, sill\u00e4 nykytilanne on t\u00e4ysin kest\u00e4m\u00e4t\u00f6n. Tilanne on nyt todella sekava ja pitk\u00e4ll\u00e4 aikav\u00e4lill\u00e4 my\u00f6s \u00e4\u00e4rimm\u00e4isen vaarallinen. Sen vuoksi kehottaisin hyv\u00e4ksym\u00e4\u00e4n tuon paketin mahdollisimman pian. Tulevaisuudessa olisi syyt\u00e4 luoda lyhyt katsaus tapaan, jolla EBRD:n varoja valvotaan. Kehottaisin EBRD:n p\u00e4\u00e4johtajaa laatimaan jotakin t\u00e4h\u00e4n liittyv\u00e4\u00e4 pankin verkkosivuille. Pankin tileist\u00e4 on huomautettava, ettei mik\u00e4\u00e4n ulkoinen tilintarkastusyritys tarkasta niit\u00e4. N\u00e4ht\u00e4v\u00e4sti ei ole mit\u00e4\u00e4n riippumatonta elint\u00e4, joka valvoisi pankin toimintaa. Tilintarkastustuomioistuin ei ole laatinut kertomusta vuosiin. Pankissa toimii arviointiyksikk\u00f6, joka laatii jonkin verran hyvinkin laadukkaita arviointeja, mutta ulkoinen tilintarkastus puuttuu. Sen vuoksi haluaisin kuulla pankin p\u00e4\u00e4johtajalta, olisiko pankki halukas avaamaan ovensa ulkoiselle tilintarkastukselle. PARSEP Arvoisa puhemies, puhun hyvin lyhyesti ja teen kolme huomautusta. Ensimm\u00e4isen teen vastatakseni j\u00e4sen Blakille. Olen valmis lis\u00e4\u00e4m\u00e4\u00e4n avoimuutta, mutta minun on my\u00f6s todettava, ett\u00e4 olemme ulkoisten tilintarkastajien valvonnan alaisia ja ett\u00e4 valvonnan tulokset ovat julkisia. Emme tyydy pelk\u00e4st\u00e4\u00e4n sis\u00e4iseen valvontaan, vaan haemme ulkopuolisten lausuntoja ja k\u00e4yt\u00e4mme tilintarkastajia, joiden lausunnot julkistetaan. My\u00f6s tilimme julkistetaan vuosittain. PARSEP EBRD:n hallintoneuvosto on erityisen tarkka tileist\u00e4\u00e4n ja k\u00e4y vuoropuhelua ulkoisten tilintarkastajien kanssa. Haluan viel\u00e4 t\u00e4sment\u00e4\u00e4, ett\u00e4 olemme tiukentaneet s\u00e4\u00e4nt\u00f6j\u00e4mme er\u00e4it\u00e4 yksityisi\u00e4 yhti\u00f6it\u00e4 ja tilintarkastajien teht\u00e4vi\u00e4 koskevien keskustelujen perusteella ja uskoakseni omaksuneet siten parempia k\u00e4yt\u00e4nteit\u00e4 t\u00e4ll\u00e4 alalla. Olemme etenkin tehneet selke\u00e4n vastuunjaon tilintarkastajien ja konsulttien v\u00e4lill\u00e4 sek\u00e4 vahvistaneet tilintarkastuksista vastaavan hallintoneuvoston erityiskomitean sek\u00e4 ulkoisten tilintarkastajien v\u00e4list\u00e4 keskustelua. PARSEP Toinen huomioni on osoitettu j\u00e4sen Schroedterille. T\u00e4\u00e4ll\u00e4 annetut esimerkit - kahdesta hankkeesta, joissa on ollut huomattavia vaikeuksia mutta my\u00f6s my\u00f6nteisi\u00e4 n\u00e4k\u00f6kohtia ja joista EBRD ei ole viel\u00e4 tehnyt p\u00e4\u00e4t\u00f6st\u00e4 - osoittavat mielest\u00e4ni melko hyvin, mihin EBRD:ll\u00e4 on mahdollisuus pyrki\u00e4: avoimuutta on lis\u00e4tt\u00e4v\u00e4, EBRD:n merkityst\u00e4 kansalaisten kannalta on lis\u00e4tt\u00e4v\u00e4 sek\u00e4 hallintoneuvoston on lopulta teht\u00e4v\u00e4 p\u00e4\u00e4t\u00f6s. Korostan viel\u00e4 kerran, ettei p\u00e4\u00e4t\u00f6ksi\u00e4 ole viel\u00e4 tehty, ja olen pannut huomautukset merkille. Toteaisin viel\u00e4, ett\u00e4 t\u00e4m\u00e4 on yksi niist\u00e4 toimista, jotka EBRD voisi mahdollisesti toteuttaa, toisin sanoen sen avulla voitaisiin lis\u00e4t\u00e4 ydinturvallisuutta, EBRD:n merkityst\u00e4 kansalaisten kannalta ja avoimuutta sek\u00e4 torjua korruptiota eri hankkeissa. T\u00e4m\u00e4 on mielest\u00e4ni aivan v\u00e4ltt\u00e4m\u00e4t\u00f6nt\u00e4. Se on valtion interventiopolitiikan hinta, ja se on saavutettavissa p\u00e4\u00e4tt\u00e4v\u00e4isell\u00e4 valtion interventiopolitiikalla. PARSEP Kolmanneksi kerron viel\u00e4 kuunnelleeni kaikkia puheenvuoroja suurella mielenkiinnolla. Minusta n\u00e4ist\u00e4 puheenvuoroista ilmeni selke\u00e4sti EBRD:n teht\u00e4v\u00e4n syv\u00e4llisyys ja voimavarat sek\u00e4 siihen liittyv\u00e4t haasteet. Moni teist\u00e4 itse asiassa puhui EBRD:n historiallisesta teht\u00e4v\u00e4st\u00e4 ja sen ajankohtaisuudesta. Minusta se on nyt t\u00e4ysin ajankohtainen ja se on hyvin ajan tasalla, kunhan toimintaa jatketaan. N\u00e4in ollen huomautuksenne edist\u00e4v\u00e4t sen s\u00e4ilymist\u00e4. T\u00e4st\u00e4 p\u00e4\u00e4senkin er\u00e4\u00e4seen meille hyvin t\u00e4rke\u00e4\u00e4n aiheeseen eli vuoropuheluun kaikkien kyseisen alueen hallitusten ja viranomaisten kanssa. PARSEP K\u00e4ymme jatkuvaa korkean tason poliittista vuoropuhelua kaikissa alueen valtioissa. Yksi haaste on minusta olennainen, nimitt\u00e4in se, ett\u00e4 oikeusvaltiota sovelletaan ei niink\u00e4\u00e4n (onneksi) ilment\u00e4m\u00e4ll\u00e4 sen lakeja vaan panemalla niit\u00e4 t\u00e4yt\u00e4nt\u00f6\u00f6n. Oikeusj\u00e4rjestelm\u00e4n ja sen toiminnan parantaminen kaikissa alueen maissa on edelleen \u00e4\u00e4rimm\u00e4isen t\u00e4rke\u00e4 haaste. Sen vuoksi muistuttaisinkin niist\u00e4 muutamista EBRD:n vakaumuksista, jotka se on omaksunut kymmenvuotisen kokemuksensa perusteella: markkinatalous ei toimi ilman demokratiaa, markkinatalous ei toimi ilman sosiaalista yhteenkuuluvuutta, sosiaalista yhteenkuuluvuutta ei ole ilman voimakasta kansalaisyhteiskuntaa, ja voimakasta kansalaisyhteiskuntaa ei ole ilman avoimuutta ja korruption torjuntaa. T\u00e4t\u00e4 sanomaa viemme toki jatkuvasti eteenp\u00e4in paitsi politiikassa, kun arvioimme eri valtioiden asemaa EBRD:n sopimuksen 1 artiklaan n\u00e4hden, my\u00f6s jatkuvassa vuoropuhelussamme. PARSEP N\u00e4in ollen haluan esitt\u00e4\u00e4 omasta puolestani ja pankin puolesta esittelij\u00e4 Markoville ja kaikille teille vilpitt\u00f6m\u00e4t kiitokset kommenteistanne. Ne antavat meille pontta jatkaa entist\u00e4kin p\u00e4\u00e4tt\u00e4v\u00e4isemmin toimintaamme, jotta voimme kahden vuoden kuluttua palata kertomaan esittelij\u00e4 Markoville sek\u00e4 talous- ja raha-asioiden valiokunnalle edistymisest\u00e4mme. Arvoisa puhemies, pahoittelen, ett\u00e4 jouduin pyyt\u00e4m\u00e4\u00e4n lis\u00e4aikaa, mutta en saanut vastausta kysymykseeni. Onko Euroopan j\u00e4lleenrakennus- ja kehityspankki siirt\u00e4nyt kaksi miljardia euroa Euratomille vai ei? Vastaukseksi riitt\u00e4\u00e4 pelkk\u00e4 kyll\u00e4 tai ei. PARSEP Esityslistalla on seuraavana Hugues Martinin laatima kalatalousvaliokunnan mietint\u00f6 (A5-0448/2002) vesiviljelyst\u00e4 Euroopan unionissa: nykyhetki ja tulevaisuus (2002/2058(INI)). PARSEP Arvoisa puhemies, t\u00e4ss\u00e4 mietinn\u00f6ss\u00e4 on monia ristiriitaisuuksia. Toisaalta siin\u00e4 on my\u00f6s er\u00e4it\u00e4 my\u00f6nteisi\u00e4 kohtia siin\u00e4 mieless\u00e4, ett\u00e4 siin\u00e4 k\u00e4sitell\u00e4\u00e4n koko vesiviljelyalaa ja kest\u00e4v\u00e4\u00e4 kehityst\u00e4 sek\u00e4 erityisesti kest\u00e4v\u00e4n kehityksen vaikutusta ymp\u00e4rist\u00f6\u00f6n. My\u00f6nteist\u00e4 siin\u00e4 on my\u00f6s se, ett\u00e4 siin\u00e4 pohditaan el\u00e4inten hyvinvoinnin eritt\u00e4in t\u00e4rke\u00e4\u00e4 kysymyst\u00e4 ja kalanviljelytilojen tilanpuuteongelmaa, sek\u00e4 kehotetaan jonkinlaisen kalatiheyden m\u00e4\u00e4ritt\u00e4miseen. PARSEP Mietinn\u00f6n toisissa kohdissa kuitenkin arvostellaan ymp\u00e4rist\u00f6\u00e4, terveytt\u00e4 ja elintarvikkeiden turvallisuutta koskevia EU:n vaatimuksia. EU n\u00e4ytt\u00e4\u00e4 panevan liiketoiminnan edut edelle, mik\u00e4 on hyvin vaarallista, sill\u00e4 ymp\u00e4rist\u00f6\u00e4 ja terveytt\u00e4 koskevien vaatimusten olisi ment\u00e4v\u00e4 liiketoiminnan etujen edelle. PARSEP Mietinn\u00f6ss\u00e4 my\u00f6s kyseenalaistetaan ennalta varautumisen periaate sill\u00e4 perusteella, ett\u00e4 se n\u00e4ytt\u00e4isi hankaloittavan vesiviljelyalan yritysten toimintaa. Meid\u00e4n on noudatettava tuota periaatetta Euroopan unionissa, ja se on otettava huomioon kaikissa tilanteissa. Se vaikeuttaa mahdollisesti asioita sen vuoksi, ett\u00e4 kyse on ennalta varautuvasta l\u00e4hestymistavasta, jonka mukaisesti meid\u00e4n olisi suhtauduttava vesiviljelyyn. PARSEP Er\u00e4s toinen mietinn\u00f6n arveluttava n\u00e4k\u00f6kohta koskee halua saada komissio edist\u00e4m\u00e4\u00e4n vesiviljely\u00e4. Minusta t\u00e4m\u00e4 ei ole hyv\u00e4 ajatus. Vesiviljelyyn liittyy monia ongelmia. Vesiviljelyalalla ty\u00f6skentelev\u00e4t ihmiset n\u00e4ytt\u00e4v\u00e4t ajattelevan, ett\u00e4 vesiviljelyll\u00e4 voidaan ratkaista luonnonvaraisiin kantoihin ja my\u00f6s merikalastukseen ja merten kalakantojen v\u00e4henemiseen liittyv\u00e4t ongelmat. T\u00e4m\u00e4 ei kuitenkaan pid\u00e4 paikkaansa. Vesiviljely synnytt\u00e4\u00e4 monia ongelmia. Muuntogeenisten kalojen ongelma on jo otettu esiin. Niiden ottaminen mukaan vesiviljelyyn on hyvin vaarallista, ja sit\u00e4 pit\u00e4isi v\u00e4ltt\u00e4\u00e4. PARSEP On my\u00f6s pohdittava teollisen kalastuksen k\u00e4ytt\u00f6\u00e4 vesiviljelyn rehutarpeiden tyydytt\u00e4miseen. On t\u00e4ysin selv\u00e4\u00e4, ett\u00e4 jos vesiviljely jatkaa nykyist\u00e4 kasvuaan, kysynt\u00e4 alalla ylitt\u00e4\u00e4 pian teollisen kalastuksen resurssitarjonnan. N\u00e4in tulee k\u00e4ym\u00e4\u00e4n, halusimmepa me sit\u00e4 tai emme. Kaiken lis\u00e4ksi vesiviljelyn vaikutukset ymp\u00e4rist\u00f6\u00f6n ovat \u00e4\u00e4rimm\u00e4isen haitalliset. PARSEP Lopuksi sanoisin viel\u00e4 kuluttajien luottamuksesta, ett\u00e4 suhtaudun hyvin varovaisesti mietinn\u00f6ss\u00e4 esitettyyn ajatukseen siit\u00e4, ett\u00e4 komission pit\u00e4isi edist\u00e4\u00e4 kuluttajien luottamusta vesiviljelyyn. Vesiviljelytuotteet puhuvat omasta puolestaan, ja meid\u00e4n olisikin nyt kiinnitett\u00e4v\u00e4 huomiota luonnonvaraisiin kalakantoihin. PARSEP Haluaisin ensinn\u00e4kin kiitt\u00e4\u00e4 esittelij\u00e4 Martinia loistavasta mietinn\u00f6st\u00e4. Vesiviljelyyn liittyv\u00e4t ongelmat ja haasteet tulivat hyvin esiin h\u00e4nen mietinn\u00f6ss\u00e4\u00e4n. Voidaan my\u00f6s todeta, ett\u00e4 Markov harjoitti omakohtaista tutkimusta k\u00e4ym\u00e4ll\u00e4 paikoissa, joissa t\u00e4t\u00e4 ammattia harjoitetaan. Tekisin kuitenkin muutamia huomioita. Yksi vesiviljelyn suurimmista ongelmista on sen huono julkisuuskuva, joka johtuu tiedonpuutteesta. Tiedonpuutteen korjaaminen onnistuu vain, jos meill\u00e4 on k\u00e4yt\u00f6ss\u00e4mme selkeit\u00e4 testitietoja. Sen vuoksi Euroopan unionin on mahdollistettava tutkimuksen harjoittaminen. Selke\u00e4t tiedot mahdollistavat alan julkisuuskuvan kohentumisen, mik\u00e4 puolestaan toisi ratkaisun alan keskeiseen ongelmaan. T\u00e4m\u00e4 ongelma on ratkaistava viestinn\u00e4n keinoin. PARSEP Rahoituksen osalta sanoisin, ett\u00e4 alan taloudellinen kannattavuus on suuri huolenaihe Euroopassa. Jos haluamme varmistaa pitk\u00e4kestoisen taloudellisen kannattavuuden, on pidett\u00e4v\u00e4 huoli siit\u00e4, ettei ala tule riippuvaiseksi tuista. T\u00e4ss\u00e4 oma apu on paras apu. L\u00e4htisin siit\u00e4, ett\u00e4 markkinoiden on annettava kulkea omaa tiet\u00e4\u00e4n ja edet\u00e4 omaa vauhtiaan, jolloin ala voisi huolehtia itse itsest\u00e4\u00e4n. Sen sijaan haluamme hallita toiminnan puitteita vastuullisesti, mink\u00e4 vuoksi meid\u00e4n on itse m\u00e4\u00e4ritelt\u00e4v\u00e4 kyseiset puitteet ja edellytykset. Markkinat on saatava oikeille raiteille. S\u00e4\u00e4telyn on oltava yksinkertaista, selke\u00e4\u00e4 ja avointa. S\u00e4\u00e4telyn esteiden poistaminen lis\u00e4\u00e4 markkinoiden yhdenmukaisuutta ja avoimuutta ja siten niiden tehokkuutta. T\u00e4m\u00e4 parantaa hinnoiltaan kilpailukykyisten tuotteiden mahdollisuuksia, mist\u00e4 my\u00f6s eurooppalaiset kuluttajat hy\u00f6tyv\u00e4t. PARSEP Kuluttajat hy\u00f6tyv\u00e4t lis\u00e4ksi elintarvikkeiden turvallisuudesta sek\u00e4 avoimuudesta, jotka ovat molemmat eritt\u00e4in t\u00e4rkeit\u00e4 n\u00e4k\u00f6kohtia. Elintarvikkeiden turvallisuus on voitava taata kaikissa oloissa. T\u00e4ss\u00e4 meid\u00e4n on kuitenkin s\u00e4ilytett\u00e4v\u00e4 suhteellisuudentaju ja unohdettava kohtuuttomat vaatimukset. Jos jokin on 100-prosenttisen hyv\u00e4\u00e4, sen ei tarvitse olla sit\u00e4 150-prosenttisesti. Jalat on pidett\u00e4v\u00e4 maassa. Kohtuuttomat vaatimukset vaikeuttaisivat pienten yritysten toimintaa, jolloin ne eiv\u00e4t pystyisi jatkamaan toimintaansa. Sen vuoksi per\u00e4\u00e4nkuulutan yksinkertaista lains\u00e4\u00e4d\u00e4nt\u00f6\u00e4, joka olisi sek\u00e4 selke\u00e4 ja toimiva ett\u00e4 sellainen, jota voidaan valvoa ja joka voidaan panna t\u00e4yt\u00e4nt\u00f6\u00f6n. ELDR-ryhm\u00e4 antaa tukensa t\u00e4lle mietinn\u00f6lle. Minusta vesiviljelyll\u00e4 voidaan my\u00f6nteisesti edist\u00e4\u00e4 luonnonvaraisten kalakantojen s\u00e4ilymist\u00e4. PARSEP Arvoisa puhemies, koska komissio on n\u00e4ht\u00e4v\u00e4sti p\u00e4\u00e4tynyt sellaiseen yhteiseen kalastuspolitiikkaan, jota on mukautettu enemm\u00e4n er\u00e4iden j\u00e4senvaltioiden lyhytaikaisten etujen kuin kalastajien tarpeiden mukaan, on rohkaisevaa huomata, ett\u00e4 se on nyt kerrankin toiminut oikein. PARSEP Yleisesti ottaen olen samaa mielt\u00e4 vesiviljelyn tulevaisuutta koskevassa komission tiedonannossa omaksutusta l\u00e4hestymistavasta. Haluan kuitenkin heti aluksi sanoa, ett\u00e4 vesiviljely on sis\u00e4llytett\u00e4v\u00e4 kokonaisuudessaan yhteiseen kalastuspolitiikkaan. Sen on voitava t\u00e4ydent\u00e4\u00e4 muita pyyntimuotoja. Sit\u00e4 ei pid\u00e4 miss\u00e4\u00e4n nimess\u00e4 pit\u00e4\u00e4 korvaavana menetelm\u00e4n\u00e4. Vesiviljely on yksi maailman nopeimmin kasvavista elintarviketuotannon aloista. Esimerkiksi Irlannin vesiviljelyalan osuus on nyky\u00e4\u00e4n yli 30 prosenttia kalastuksen kokonaistuotannosta, mik\u00e4 vastaa yleismaailmallista suuntausta. Vesiviljelyll\u00e4 on suuret mahdollisuudet luoda lis\u00e4\u00e4 ty\u00f6paikkoja ja lis\u00e4t\u00e4 taloudellista toimintaa etenkin rannikkoyhteis\u00f6iss\u00e4, joissa ty\u00f6paikkoja on hyvin v\u00e4h\u00e4n. Meren tuotteiden maailmanlaajuisen kysynn\u00e4n vuoksi vesiviljelyn merkitys raaka-aineen l\u00e4hteen\u00e4 kalanjalostusteollisuudessa kasvaa jatkuvasti, koska se tuo huomattavaa lis\u00e4arvoa ja vientimahdollisuuksia. On selv\u00e4\u00e4, ett\u00e4 kaikki alan tuleva kehitt\u00e4minen on perustettava kest\u00e4v\u00e4\u00e4n kehitykseen. Koko Euroopan unionissa on pyritt\u00e4v\u00e4 kalastusalalla l\u00f6yt\u00e4m\u00e4\u00e4n ensisijaisesti sellaisia parhaita k\u00e4yt\u00e4nt\u00f6j\u00e4, jotka ovat sopusoinnussa tiukkojen ymp\u00e4rist\u00f6alan suuntaviivojen ja korkeiden laatuvaatimusten kanssa, kuten nyt on tehty Irlannissa. PARSEP Lis\u00e4ksi alalla on sovellettava parhaita k\u00e4yt\u00e4nt\u00f6j\u00e4 ottamalla huomioon elintarvikkeiden turvallisuus, ja t\u00e4lt\u00e4 osin olen eri mielt\u00e4 irlantilaisen kollegani McKennan kanssa. Vesiviljely ei ole haitallista, vaan se on luonnonmukainen ja luonnonvaroihin perustuva ala, joka on kaikkea muuta kuin haitallinen. EU:n ulkopuolisten tuottajien aiemmin harjoittama ulkomaisten kalastustuotteiden, etenkin lohen, polkumyynti saattoi yhteis\u00f6n tuottajat hyvin ep\u00e4edulliseen asemaan, ja tekee niin edelleen. T\u00e4ss\u00e4 yhteydess\u00e4 kehottaisinkin komissiota miettim\u00e4\u00e4n uudelleen sen ehdottamiin toimiin sis\u00e4ltyv\u00e4\u00e4 julkisten tukien osaa, ennen kuin se esitt\u00e4\u00e4 muodolliset ehdotuksensa. Rehev\u00f6itymisen osalta panen merkille komission varovaisen suhtautumisen maatalousalan direktiiviin veden saastumisesta. Vesiviljely ja maatalous ovat toisistaan t\u00e4ysin poikkeavia aloja, joten niit\u00e4 on k\u00e4sitelt\u00e4v\u00e4 erikseen. Sen vuoksi tarvitsemme mielest\u00e4ni erityisen direktiivin kaikista mahdollisista saastumisen muodoista vesiviljelyn alalla. Kiit\u00e4n teit\u00e4, kollega Martin, t\u00e4st\u00e4 t\u00e4rke\u00e4st\u00e4 aiheesta laatimastanne ensiluokkaisesta mietinn\u00f6st\u00e4, ja toivon, ett\u00e4 kalatalousvaliokunnassa sille annettu yksimielinen tuki l\u00f6ytyisi my\u00f6s t\u00e4\u00e4lt\u00e4 parlamentista t\u00e4n\u00e4\u00e4n. PARSEP Arvoisa puhemies, t\u00e4m\u00e4 on hyv\u00e4, vankka ja hyvin laadittu mietint\u00f6, jossa k\u00e4sitell\u00e4\u00e4n t\u00e4rke\u00e4\u00e4 alaa, jolla meill\u00e4 on viel\u00e4 paljon teht\u00e4v\u00e4\u00e4, sill\u00e4 tuomme yli puolet kuluttamastamme kalasta muualta. Kyse on alasta, jolla j\u00e4ljitett\u00e4vyys varmistetaan parhaiten kotimaisen tuotannon avulla ja jolla vesiviljely tuo meille lis\u00e4\u00e4 elint\u00e4rkeit\u00e4 ty\u00f6paikkoja rannikkoalueillemme, sill\u00e4 kalavarathan ovat rajalliset. PARSEP Esittelij\u00e4 on oikeassa tuomitessaan sen, ettei vihre\u00e4ss\u00e4 kirjassa mainita vesiviljely\u00e4, ja korostaessaan vuoden 2002 syyskuussa annetun komission tiedonannon puutteita. Vesiviljelyalalle on erityisen tyypillist\u00e4 tutkimustoimien merkitys ja kesto. N\u00e4it\u00e4 toimia ei ole aina suhteutettu mahdollisuuksiin my\u00f6nt\u00e4\u00e4 rahoitusta nuorille perheyrityksille, joilla on v\u00e4h\u00e4n p\u00e4\u00e4omaa ja jotka joutuvat toimimaan kiivaassa kilpailuymp\u00e4rist\u00f6ss\u00e4, jossa hinnat vaihtelevat jatkuvasti. Esimerkiksi er\u00e4s ranskalainen yritys, joka oli aloittamassa piikkikampelan tuotantoa Euroopassa ja jonka johtajana toimii Vend\u00e9en alueelta kotoisin oleva Michel Adrien, joutui investoimaan 2,5 miljoonaa frangia joka vuosi kymmenen vuoden ajan pelkk\u00e4\u00e4n tutkimukseen, ennen kuin se sai kehitetty\u00e4 menetelm\u00e4n vankeudessa el\u00e4vien piikkikampeloiden lis\u00e4\u00e4ntymist\u00e4 varten. PARSEP Voidaanko ja pit\u00e4isik\u00f6 ajatella, ett\u00e4 vain yksityiset yritykset vastaisivat t\u00e4m\u00e4ntyyppisist\u00e4 tutkimusohjelmista, jotka ovat hyvin kalliita, aikaa vievi\u00e4 sek\u00e4 teknisesti ja kaupallisesti hyvin ep\u00e4varmoja, ja joissa k\u00e4sitell\u00e4\u00e4n biologisia peruskysymyksi\u00e4? Minusta ei. P\u00e4invastoin olen sit\u00e4 mielt\u00e4, ett\u00e4 suurin osa t\u00e4llaisesta tutkimuksesta on teht\u00e4v\u00e4 julkisin varoin samoin kuin pienkalastusalusten jatkuvaa uudistamista on rahoitettava julkisista varoista. On t\u00e4ysin selv\u00e4\u00e4, ettei julkisen rahoituksen voida t\u00e4ss\u00e4 tapauksessa katsoa aiheuttavan kilpailun v\u00e4\u00e4ristymist\u00e4. PARSEP Tutkimuksesta puhuttaessa tarkoitan etenkin sellaisten uusien rokotteiden tutkimusta, joilla voidaan poistaa antibioottien k\u00e4ytt\u00f6\u00f6n liittyv\u00e4t riskit. T\u00e4m\u00e4 on ratkaiseva kysymys tulevaisuuden kannalta ja sen suhteen, miten t\u00e4m\u00e4 ala k\u00e4sitet\u00e4\u00e4n. N\u00e4in ollen on luotava voimakkaita kannustimia sek\u00e4 j\u00e4senvaltioissa ett\u00e4 yhteis\u00f6ss\u00e4. Kannatan my\u00f6s ajatusta KOR:n s\u00e4\u00e4nt\u00f6jen mukauttamisesta, ja minusta my\u00f6s EMOTR:n s\u00e4\u00e4nt\u00f6j\u00e4 pit\u00e4isi mukauttaa, jotta voitaisiin rahoittaa nuorten yritt\u00e4jien perustamia yrityksi\u00e4, alalle siirtymist\u00e4 ja kuluttajille tiedottamista. PARSEP Yhdyn esittelij\u00e4n huoleen muuntogeenisen kalan riskeist\u00e4. On t\u00e4ysin selv\u00e4\u00e4, ett\u00e4 Euroopan unionissa on kiellett\u00e4v\u00e4 kokonaan muuntogeenisen kalan tuotanto ja maahantuonti. Lis\u00e4ksi on oltava hyvin tarkkana monien kolmansien maiden harjoittaman ep\u00e4reilun kilpailu suhteen, sill\u00e4 hintojen polkumyynti on ehdottomasti estett\u00e4v\u00e4 n\u00e4ill\u00e4 kasvavilla ja ep\u00e4vakailla markkinoilla. On ehdottomasti kiellett\u00e4v\u00e4 sellaisten vesiviljelytuotteiden maahantuonti, jotka eiv\u00e4t ole Euroopan unionin tuottajille asetettujen vaatimusten mukaisia. T\u00e4m\u00e4n kiellon noudattamista on my\u00f6s valvottava asianmukaisesti. PARSEP Arvoisa puhemies, haluan aloittaa onnittelemalla esittelij\u00e4 Martinia h\u00e4nen loistavasta aloitteestaan, joka edustaa uutta n\u00e4kemyst\u00e4 vesiviljelyalueiden luomisesta ja vesiviljelyst\u00e4. Kuten kaikki tied\u00e4mme, komissio on yhteisen kalastuspolitiikan uudistuksen yhteydess\u00e4 tehnyt perinpohjaisia muutoksia kalavarojen s\u00e4\u00e4telymenetelmiin ja kalastuslaivaston uudistusmenetelmiin. Komissio on erityisesti esitt\u00e4nyt kalastuslaivastojen uudistamiseen suunnatun tuen poistamista, sen nykyaikaistamiseen suunnatun tuen v\u00e4hent\u00e4mist\u00e4 sek\u00e4 sellaisen laivastojen hallinnointipolitiikan harjoittamista, joka heikent\u00e4isi rakenneuudistuksen toteuttamiskelpoisuutta jopa Portugalin kaltaisissa maissa, joissa tavoitteita ei ole vain t\u00e4ytetty vaan jopa ylitetty. T\u00e4llainen toimenpide on vaikuttanut rajusti kalastuslaivastojen kokoon ja siten v\u00e4hent\u00e4nyt niiden toimintaa. T\u00e4llaisen toimintamenetelm\u00e4n vaikutukset ovat vaatineet - ja vaativat jatkossakin - todellisia uhrauksia Portugalin rannikkoyhteis\u00f6ilt\u00e4, jotka ovat vuosikymmeni\u00e4 saaneet elantonsa merest\u00e4 ja el\u00e4neet meren antimilla. PARSEP N\u00e4in ollen puhun t\u00e4\u00e4ll\u00e4 t\u00e4ysistunnossa edell\u00e4 mainittujen ihmisten ja kaikkien merest\u00e4 elantonsa hankkivien eurooppalaisten puolesta. Heid\u00e4n vuokseen pyrin tukemaan ratkaisuja, joilla voitaisiin minimoida yhteis\u00f6n toimenpiteiden taloudelliset, sosiaaliset ja kulttuuriset seuraukset. N\u00e4m\u00e4 toimenpiteet ovat hyvi\u00e4 kalavarojen kannalta, mutta niiss\u00e4 n\u00e4k\u00f6j\u00e4\u00e4n v\u00e4heksyt\u00e4\u00e4n sek\u00e4 kalastusta ett\u00e4 siit\u00e4 aina elantonsa saaneita ihmisi\u00e4. Uskon, ett\u00e4 vesiviljelyll\u00e4 voidaan jossakin m\u00e4\u00e4rin t\u00e4ydent\u00e4\u00e4 perinteist\u00e4 kalastusta, jolloin on mahdollista hy\u00f6dynt\u00e4\u00e4 kalastustoiminnassa aina mukana olleiden kokemuksia ja palauttaa n\u00e4iden samalla toiminnanalalla ty\u00f6skentelevien ihmisten ammatillinen arvostus. PARSEP Alan kasvava merkitys edellytt\u00e4\u00e4 unionilta vakaita investointeja sek\u00e4 tieteelliseen tutkimukseen ett\u00e4 koulutukseen ja ammattip\u00e4tevyyteen, j\u00e4rjestelmien ja laitteistojen k\u00e4ytt\u00f6\u00f6nottoon ja uudistamiseen sek\u00e4 terveydensuojeluun. T\u00e4ss\u00e4 yhteydess\u00e4 olisi ennen kaikkea edistett\u00e4v\u00e4 laadukkaiden kalajauhojen ja \u00f6ljyjen k\u00e4ytt\u00f6\u00e4, rajoitettava antibioottien k\u00e4ytt\u00f6\u00e4 ja torjuttava muuntogeenisen kalan k\u00e4ytt\u00f6\u00e4. PARSEP Yhteis\u00f6ss\u00e4 on annettu noin 150 asetusta vesiviljelyst\u00e4. Todellista oikeudellista suojaa ei alalla kuitenkaan varmisteta lukuisten asetusten avulla, vaan siten, ett\u00e4 parannetaan yhteis\u00f6n lains\u00e4\u00e4d\u00e4nn\u00f6n ja toimien m\u00e4\u00e4rittelyn laatua. T\u00e4t\u00e4 voidaan parantaa toteuttamalla ja kohdistamalla oikein yhteis\u00f6n investointeja sek\u00e4 yksinkertaistamalla lains\u00e4\u00e4d\u00e4nt\u00f6\u00e4 siten, ett\u00e4 siit\u00e4 tehd\u00e4\u00e4n kohderyhm\u00e4n silmiss\u00e4 entist\u00e4 yhdenmukaisempi ja ymm\u00e4rrett\u00e4v\u00e4mpi, jolloin sit\u00e4 voidaan panna entist\u00e4 tehokkaammin t\u00e4yt\u00e4nt\u00f6\u00f6n. PARSEP N\u00e4m\u00e4 ovat mielest\u00e4ni ratkaisevan t\u00e4rkeit\u00e4 n\u00e4k\u00f6kohtia sen v\u00e4ltt\u00e4m\u00e4tt\u00f6m\u00e4n muutoksen kannalta, joka on tapahduttava asenteessa ja julkisuudenkuvassa, jotka jo nyt vahingoittavat k\u00e4sityst\u00e4, joka kansalaisilla ja etenkin kuluttajilla on t\u00e4st\u00e4 toiminnanalasta tiettyjen kalalajien osalta. Kaikki ovat varmasti samaa mielt\u00e4 siit\u00e4, ett\u00e4 KOR-asetusta on muutettava, jotta voitaisiin tukea nuorten vesiviljelytilojen perustamista, pieni\u00e4 ja keskisuuria yrityksi\u00e4, kalataloustuotteiden laadun edist\u00e4mist\u00e4, n\u00e4iden tuotteiden kulutuksen edist\u00e4miskampanjoita, ymp\u00e4rist\u00f6yst\u00e4v\u00e4llisten k\u00e4yt\u00e4nt\u00f6jen k\u00e4ytt\u00f6\u00f6nottoa ja niin edelleen. Esitt\u00e4isin n\u00e4in ollen komissiolle ja komission j\u00e4senelle Solbes Miralle - harmi, ettei komission j\u00e4sen Franz Fischler ole l\u00e4sn\u00e4 t\u00e4\u00e4ll\u00e4 parlamentissa - vetoomuksen siit\u00e4, ettei t\u00e4m\u00e4 aihe j\u00e4isi unohduksiin ja ett\u00e4 toteutettaisiin tehokkaita toimia vesiviljelyalan kehitt\u00e4miseksi entisest\u00e4\u00e4n ja tulevaisuutta silm\u00e4ll\u00e4 pit\u00e4en. PARSEP Arvoisa puhemies, arvoisa komission j\u00e4sen, 20 vuotta sitten vesiviljely\u00e4 pidettiin ratkaisuna kalansaaliiden hiipumiseen. Toivottiin, ett\u00e4 se voisi t\u00e4ytt\u00e4\u00e4 suuren osan n\u00e4lk\u00e4alueilla asuvien ihmisten proteiinin tarpeesta. Monien toiveiden tavoin t\u00e4m\u00e4k\u00e4\u00e4n ei k\u00e4ynyt toteen. Viljellyn kalan kohdalla ravinnon muuntamisaste on alhainen. Yhteen kalakiloon ei tarvita nelj\u00e4\u00e4 kiloa kalarehua, kuten nis\u00e4kk\u00e4iden kohdalla, vaan vain 1,5 - 2 kiloa. Valitettavasti suuri osa t\u00e4st\u00e4 kalarehusta on valmistettava el\u00e4inproteiinista. Kalarehun proteiini on per\u00e4isin kalajauhosta, joka puolestaan on per\u00e4isin teollisesta pyynnist\u00e4. Pyynti tapahtuu er\u00e4\u00e4nlaisilla nailonverkoilla, jotka tekev\u00e4t paljaiksi laajoja alueita. Niill\u00e4 ei pyydet\u00e4 vain t\u00e4ysikasvuista kalaa vaan my\u00f6s alamittaista kalaa. My\u00f6s muun muassa turskan verta k\u00e4sitell\u00e4\u00e4n teollisesti. N\u00e4in ollen ei ole mielt\u00e4 puhua turskan elvytyssuunnitelmista, jos ei ensin rajoiteta teollista kalastusta. Sit\u00e4 voidaan rajoittaa kiinti\u00f6in, mutta my\u00f6s pakottamalla kalajauhon ostajat harkitsemaan vaihtoehtoisia raaka-aineita, kuten kasviproteiinia tai el\u00e4inplanktonia. N\u00e4m\u00e4 vaihtoehtoiset raaka-materiaalit eiv\u00e4t ole k\u00e4sitt\u00e4\u00e4kseni valmiiksi saatavilla. Sen vuoksi on hyvin t\u00e4rke\u00e4\u00e4 tarjota taloudellisia kannustimia n\u00e4iden vaihtoehtojen tutkimukseen ja edist\u00e4\u00e4 tutkimusta aina kuin se on mahdollista. T\u00e4st\u00e4 hy\u00f6tyisiv\u00e4t paitsi Pohjanmeren turskakannat my\u00f6s maailman n\u00e4lk\u00e4\u00e4n\u00e4kev\u00e4t. Jos tutkimuksella voidaan tarjota vaihtoehtoisia proteiininl\u00e4hteit\u00e4, vesiviljelykseen liittyv\u00e4t toiveet saattaa sittenkin k\u00e4yd\u00e4 toteen. PARSEP Arvoisa puhemies, min\u00e4kin olen kiitollinen esittelij\u00e4lle, sill\u00e4 minusta voimme mietinn\u00f6n ansiosta asettaa joitakin raameja. Ensinn\u00e4kin vesiviljely tarjoaa olennaisen lis\u00e4tulojen l\u00e4hteen varsinkin syrj\u00e4isill\u00e4 alueilla asuville ja ty\u00f6skenteleville kalastajille. Toiseksi sill\u00e4 voidaan v\u00e4hent\u00e4\u00e4 kalastusta, ja se s\u00e4\u00e4st\u00e4\u00e4 kalavarojamme. Muutamia ongelmia on kuitenkin edelleen, ja ne liittyv\u00e4t pienimuotoiseen kalastukseen ja perinteiseen kalastukseen. Siirtymist\u00e4 vesiviljelyyn olisi tuettava, koska se tulee kalliiksi. Alkuinvestoinnit ja kolmesta ensimm\u00e4isest\u00e4, l\u00e4hes aina tuottamattomasta vuodesta selviytyminen edellytt\u00e4\u00e4 taloudellisia resursseja. Kalatalouden ohjauksen rahoitusv\u00e4lineeseen olisi sis\u00e4llytett\u00e4v\u00e4 erityisi\u00e4, kohdennettuja ja nopeita luottoja tai muutoin vesiviljelyst\u00e4 tulee vain varakkaiden mahdollisuus. Komission tavoite, jonka mukaan alalle olisi tarkoitus saada 8 000-10 000 ty\u00f6paikkaa lis\u00e4\u00e4, on kunnianhimoinen haaste, joka edellytt\u00e4\u00e4 konkreettista sitoumusta vesiviljelyn taloudellisen kannattavuuden edist\u00e4miseen. PARSEP Toinen tavoite - johon parlamentti kiinnitt\u00e4\u00e4 erityist\u00e4 huomiota - koskee elintarvikkeiden turvallisuutta, ymp\u00e4rist\u00f6vaikutuksia ja el\u00e4inten terveytt\u00e4. T\u00e4ss\u00e4 tarvitaan k\u00e4yt\u00e4nnes\u00e4\u00e4nt\u00f6j\u00e4 ja hyvin m\u00e4\u00e4riteltyj\u00e4 s\u00e4\u00e4nt\u00f6j\u00e4 ja rajoituksia. Ennen kaikkea on varmistettava, ett\u00e4 kaikki kolmansista maista tulevat tuotteet vastaavat hygieniaa, elintarvikkeiden turvallisuutta ja el\u00e4inten hyvinvointia koskevia vaatimuksiamme. T\u00e4ll\u00e4 ei pyrit\u00e4 vain ehk\u00e4isem\u00e4\u00e4n ep\u00e4reilua kilpailua, kuten voitaisiin ajatella, vaan ennen kaikkea parantamaan aidosti kest\u00e4v\u00e4n kehityksen tasoa. PARSEP Arvoisa puhemies, keskustellessamme t\u00e4m\u00e4n asiakirjan kompromissiversiosta kaikki ryhm\u00e4t arvelivat, ett\u00e4 pohjoista ulottuvuutta koskevan l\u00e4hestymistavan muuttuminen laajentumisen my\u00f6t\u00e4 aiheuttaa muutoksia aluepolitiikkaan. T\u00e4m\u00e4n vuoksi haluamme lis\u00e4t\u00e4 3 kohdan alkuun seuraavan kattavan kappaleen: ?on muokattava yleist\u00e4 l\u00e4hestymistapaa muuttuvassa tilanteessa, jossa kumppanivaltioista tulee EU:n j\u00e4senvaltioita; on pantava merkille asiasta aiheutuvat erityistarpeet alueellisen kehityksen ja rajatylitt\u00e4v\u00e4n yhteisty\u00f6n suhteen?. Arvoisa puhemies, ajatus on hyv\u00e4 ja se sis\u00e4ltyy jo yhteisesti hyv\u00e4ksyttyyn p\u00e4\u00e4t\u00f6slauselmaesitykseen, eli sit\u00e4 ei tarvitse en\u00e4\u00e4 t\u00e4nne lis\u00e4t\u00e4. Siksi emme tue t\u00e4t\u00e4 suullista tarkistusta. PARSEP Hyv\u00e4 parlamentin j\u00e4sen Myller, vastustatte tarkistusta, mutta jollei v\u00e4hint\u00e4\u00e4n 32 parlamentin j\u00e4sent\u00e4 vastusta tarkistusta, siit\u00e4 \u00e4\u00e4nestet\u00e4\u00e4n. PARSEP (Parlamentti hyv\u00e4ksyi p\u00e4\u00e4t\u00f6slauselman.) PARSEP Helmuth Markovin laatima talous- ja raha-asioiden valiokunnan mietint\u00f6 Euroopan j\u00e4lleenrakennus- ja kehityspankin (EBRD) toiminnasta (2002/2095(INI)) (A5-0421/2002) PARSEP Tarkistus 6: PARSEP Arvoisa puhemies, haluan esitt\u00e4\u00e4 kollegoitteni suostumuksella suullisen tarkistuksen, joka koskee tarkistusta 6. Haluamme tehd\u00e4 toiselle riville lis\u00e4yksen, joka koskee sellaisia asioita, joita emme halua pankin rahoittavan. Suullinen tarkistuksemme koskee ainoastaan ydinalan laajentamista ja kuuluu n\u00e4in: suljettava ydinalan laajentaminen erityisesti... PARSEP Olemme juuri \u00e4\u00e4nest\u00e4neet Libanonin kanssa teht\u00e4v\u00e4n assosiaatiosopimuksen puolesta. Parlamentti pyysi alle vuosi sitten Euroopan yhteis\u00f6n ja Israelin v\u00e4lisen assosiaatiosopimuksen purkamista. Mill\u00e4 perusteella se, mik\u00e4 p\u00e4ti jokin aika sitten Israeliin, ei p\u00e4de t\u00e4ll\u00e4 hetkell\u00e4 Libanoniin? PARSEP Mielest\u00e4ni olisi pidett\u00e4v\u00e4 mieless\u00e4, ettei Libanon ole demokraattinen valtio, ettei siell\u00e4 noudateta oikeusvaltioperiaatetta ja ett\u00e4 meid\u00e4n niin t\u00e4rkein\u00e4 pit\u00e4mi\u00e4mme perusoikeuksia ja periaatteita rikotaan Libanonissa p\u00e4ivitt\u00e4in. Lis\u00e4ksi Libanon on edelleen terroristij\u00e4rjest\u00f6jen, erityisesti Hizbollahin, tyyssija, ja Syyrian miehitt\u00e4m\u00e4 ja \"nujertama\". Kuitenkin olemme luomassa yhteyksi\u00e4 Libanoniin. PARSEP Mielest\u00e4ni tasapainoinen eurooppalainen politiikka olisi luonnollisesti politiikkaa, joka on avoinna todella kaikille Keski-id\u00e4n valtioille mutta jonka l\u00e4ht\u00f6kohtana on selv\u00e4sti oltava Israelin kanssa tehdyn assosiaatiosopimuksen yll\u00e4pit\u00e4minen. Monet, erityisesti tieteellisten pakotteiden k\u00e4ytt\u00f6\u00f6nottoa koskevat aloitteet, jotka ovat h\u00e4pe\u00e4llisi\u00e4 ja jotka pit\u00e4isi tuomita, uhkaavat kyseist\u00e4 sopimusta selv\u00e4sti p\u00e4ivitt\u00e4in. Mielest\u00e4ni terve eurooppalainen politiikka on Libanonille avointa politiikkaa, mutta Euroopan on pysytt\u00e4v\u00e4 my\u00f6s Israelin rinnalla. PARSEP Arvoisa puhemies, k\u00e4vin muutama p\u00e4iv\u00e4 sitten tulevassa j\u00e4senvaltiossamme Kyproksessa, joka on Ateenan sek\u00e4 Irakin ja Kuwaitin v\u00e4lisen rajan puoliv\u00e4liss\u00e4. Libanon on vain kivenheiton p\u00e4\u00e4ss\u00e4 sielt\u00e4. PARSEP T\u00e4m\u00e4 osoittaa, kuinka t\u00e4rke\u00e4 Libanon on Euroopan unionille. Ottaen huomioon, ett\u00e4 Beirut oli aikaisemmin it\u00e4isen V\u00e4limeren kulttuurinen ja taloudellinen keskus, vakaa Libanon olisi meid\u00e4n eurooppalaisten kannalta hyvin t\u00e4rke\u00e4 asia. T\u00e4m\u00e4n vuoksi pid\u00e4n t\u00e4m\u00e4n sopimuksen hyv\u00e4ksymist\u00e4 hyvin t\u00e4rke\u00e4n\u00e4, ja mielest\u00e4ni sill\u00e4 edistet\u00e4\u00e4n merkitt\u00e4v\u00e4sti Euroopalle hyvin t\u00e4rke\u00e4n maan vakauttamista. Libanonin kulttuurin moninaisuus ja sen j\u00e4lleen hitaasti kehittym\u00e4ss\u00e4 oleva taloudellinen mahti ovat meille t\u00e4rkeit\u00e4. PARSEP Mielest\u00e4ni on Euroopan etujen mukaista tukea p\u00e4\u00e4ministeri Haririn kehitt\u00e4mispyrkimyksi\u00e4, ja mielest\u00e4ni meid\u00e4n olisi perustettava Euro-V\u00e4limeri-edustajakokous Beirutiin osoitukseksi siit\u00e4, ett\u00e4 tunnustamme n\u00e4m\u00e4 pyrkimykset ja tuemme niit\u00e4, jotta maata kiistatta vaivaavat terrorismi- ja miehitysongelmat voidaan ratkaista. PARSEP Hyv\u00e4 j\u00e4sen Posselt, mielest\u00e4ni on viel\u00e4 liian aikaista mietti\u00e4 Euroopan seuraavan parlamentaarisen edustajakokouksen sijaintia. Meid\u00e4n on syyt\u00e4 odottaa hieman, ennen kuin omaksumme tietyn asenteen tai teemme asiaa koskevia p\u00e4\u00e4t\u00f6ksi\u00e4. PARSEP - (FR) Berliinin muurin sortumisen j\u00e4lkeen perustettu Euroopan j\u00e4lleenrakennus- ja kehityspankki EBRD pyrkii tukemaan taloudellisesti l\u00e4nsimaisen p\u00e4\u00e4oman vienti\u00e4 Keski- ja It\u00e4-Eurooppaan sek\u00e4 IVY-maihin eli entiseen Neuvostoliittoon. Esittelij\u00e4 ei kyseenalaista mitenk\u00e4\u00e4n EBRD:n toimia tai markkinatilannetta, jossa ne toteutetaan, vaikka h\u00e4n haluaisikin, ett\u00e4 EBRD kiinnitt\u00e4isi enemm\u00e4n huomiota ty\u00f6llisyyteen ja toimiensa taloudellisiin ja v\u00e4est\u00f6\u00f6n kohdistuviin vaikutuksiin. Erityisesti monikansallisten yhti\u00f6iden, jotka ovat, kuten esittelij\u00e4 huomauttaa, EBRD:n huomion kohteena, hy\u00f6dyntavoittelu on kuitenkin kohtalokasta v\u00e4est\u00f6lle ja sen hyvinvoinnille. PARSEP Meit\u00e4 ei ehk\u00e4 liikuta se, ett\u00e4 EBRD investoi p\u00e4\u00e4omaa Ven\u00e4j\u00e4ll\u00e4 sellaisille aloille, joilla pankit ja yksityisyritykset eiv\u00e4t halua ottaa riskej\u00e4, ja ett\u00e4 EBRD:lle on n\u00e4in ollen aiheutunut ajoittain raskaitakin tappioita siit\u00e4, ett\u00e4 se on monien paikallisten ja kansainv\u00e4listen keinottelijoiden tavoin kokeillut onneaan keinottelemalla, mik\u00e4 on aiheuttanut t\u00e4lle maalle entist\u00e4 enemm\u00e4n vahinkoa. Pantakoon kuitenkin merkille, ett\u00e4 kyseisten toimien tarkastelu paljastaa j\u00e4rkytt\u00e4v\u00e4n m\u00e4\u00e4r\u00e4n tapauksia, joissa It\u00e4-Euroopan kansoja on k\u00e4ytetty taloudellisesti hyv\u00e4ksi. PARSEP EBRD:n todellista roolia saatetaan pyrki\u00e4 salailemaan, mutta pankki on imperialistinen v\u00e4line, jota k\u00e4ytet\u00e4\u00e4n It\u00e4-Euroopan kansojen ry\u00f6v\u00e4\u00e4miseen. T\u00e4m\u00e4n vuoksi hylk\u00e4simme mietinn\u00f6n. PARSEP Onnittelen kollegaani esittelij\u00e4 Markovia h\u00e4nen eritt\u00e4in hyv\u00e4st\u00e4 mietinn\u00f6st\u00e4\u00e4n, jonka asianomainen parlamentin valiokunta hyv\u00e4ksyi yksimielisesti. EBRD ei liity Euroopan unioniin. Se ei ole my\u00f6sk\u00e4\u00e4n Maailmanpankkiryhm\u00e4n alainen kehityspankki. Sen p\u00e4\u00e4oma on per\u00e4isin yli 60 valtiolta, jotka ovat kiinnostuneita Berliinin muurin sortumisen j\u00e4lkeisest\u00e4 Keski- ja It\u00e4-Euroopan tulevaisuudesta. EBRD on joutunut vastaamaan moniin haasteisiin kahdentoista siirtym\u00e4vuoden aikana. Sen p\u00e4\u00e4johtaja Jacques de Larosi\u00e8re kuvaili meille pankin toimintaa GUE/NGL:ss\u00e4. PARSEP Katson esittelij\u00e4n tavoin, ett\u00e4 meid\u00e4n on tuettava EBRD:t\u00e4. Vaikka se pyrkii kehitt\u00e4m\u00e4\u00e4n markkinataloutta n\u00e4iss\u00e4 entisiss\u00e4 kommunistivaltioissa, sosiaalisia, ymp\u00e4rist\u00f6\u00f6n liittyvi\u00e4 ja julkisiin palveluihin liittyvi\u00e4 n\u00e4k\u00f6kohtia ei ole todellakaan sivuutettu. EBRD ei ole \u00e4\u00e4riliberalismin l\u00e4hettil\u00e4s, vaikka jotkut saattavat luulla niin. Se on valtuutettu investoimaan kohteisiin, joihin investoimista muut pankit arastelevat. Olen hyvin iloinen siit\u00e4, ett\u00e4 EBRD keskittyy toiminnassaan pienten ja keskisuurten yritysten kehitt\u00e4miseen, sill\u00e4 niiden merkitys uusien ty\u00f6paikkojen luojana on hyvin tunnettu. PARSEP \u00c4\u00e4nestin Markovin mietinn\u00f6n puolesta, koska mielest\u00e4ni mietinn\u00f6ss\u00e4 Euroopan j\u00e4lleenrakennus- ja kehityspankille annettu tunnustus, joka koskee sen pk-yrityksille It\u00e4- ja Keski-Euroopan entisiss\u00e4 kommunistivaltioissa tarjoamaa tukea, on aiheellinen. Vaikka t\u00e4m\u00e4 ty\u00f6 j\u00e4tet\u00e4\u00e4n usein mainitsematta, se on ratkaisevan t\u00e4rke\u00e4\u00e4 taloudellisten valmiuksien parantamisessa maissa, joiden pk-sektori on usein eritt\u00e4in alikehittynyt. PARSEP Arvostan erityisesti sit\u00e4, ett\u00e4 mietinn\u00f6ss\u00e4 pyydet\u00e4\u00e4n tukemaan kansalaisj\u00e4rjest\u00f6j\u00e4, jotka osallistuvat demokratian lujittamiseen Valko-Ven\u00e4j\u00e4ll\u00e4, Ukrainassa ja Moldovassa. PARSEP T\u00e4m\u00e4n mietinn\u00f6n etuna on, ett\u00e4 se muistuttaa meit\u00e4 EBRD:n t\u00e4rke\u00e4st\u00e4 teht\u00e4v\u00e4st\u00e4, joka on siirtym\u00e4vaiheessa olevien valtioiden demokratiaa lujittavan markkinatalouden luominen. T\u00e4m\u00e4n teht\u00e4v\u00e4n hoitamiseksi EBRD joutuu usein ottamaan riskej\u00e4, eik\u00e4 se voi j\u00e4rjestelm\u00e4llisesti hankkia valtion takauksia. T\u00e4m\u00e4n vuoksi se investoi my\u00f6s kohteisiin, joihin investoimista muut pankit arastelevat. EBRD on keskitt\u00e4nyt toimiaan laajasti pienten ja keskisuurten yritysten kehitt\u00e4miseen - toisaalta paikallispankeille tarkoitettujen tukiohjelmien kehitt\u00e4miseen ja toisaalta pk-yritysten luotottamiseen ja mikroluottojen tarjoamiseen erikoistuvien pankkien perustamiseen. PARSEP Panimme tyytyv\u00e4isin\u00e4 merkille, ett\u00e4 esittelij\u00e4 pit\u00e4\u00e4 EBRD:n toimintaa yleisesti ottaen tyydytt\u00e4v\u00e4n\u00e4 huolimatta t\u00e4m\u00e4n esitt\u00e4mist\u00e4 lausunnoista ja varoituksista, jotka koskevat oletettuja \u00e4\u00e4riliberalismin ja hallitsemattoman liberalisoinnin aatteita, joiden kukaan ei halua ohjaavan t\u00e4m\u00e4n laitoksen toimintaa. PARSEP N\u00e4m\u00e4 ovat lyhyesti ne syyt, joiden vuoksi tuen mietint\u00f6\u00e4. PARSEP \u00c4\u00e4nestin t\u00e4m\u00e4n mietinn\u00f6n puolesta, koska yhdyn erityisesti niihin n\u00e4kemyksiin, joita siin\u00e4 esitet\u00e4\u00e4n sen my\u00f6nteisen vaikutuksen osalta, joka EBRD:n toiminnalla on ollut Keski- ja It\u00e4-Euroopan valtioiden talouksiin. N\u00e4m\u00e4 taloudet ovat eritt\u00e4in heikkoja, eik\u00e4 niill\u00e4 ole markkina- ja kilpailutalouden mekanismeja, tapoja ja kulttuuria. Itse asiassa laaja-alaisella p\u00e4\u00e4omamarkkinoiden vapauttamisella ja sill\u00e4, ett\u00e4 hallintoelimet hyv\u00e4ksyv\u00e4t yksityiset aloitteet dynaamiseksi kehitysvoimaksi, on ollut er\u00e4it\u00e4 kielteisi\u00e4 ja huolestuttavia vaikutuksia. Esittelij\u00e4 erittelee n\u00e4m\u00e4 vaikutukset, jotka on syyt\u00e4 ottaa huomioon, seuraavasti: kleptokratian eli varasvalta ja v\u00e4est\u00f6n k\u00f6yhtyminen. T\u00e4m\u00e4n vuoksi hyv\u00e4ksyn ehdotetun ratkaisun, jonka mukaisesti n\u00e4iss\u00e4 valtioissa kehitet\u00e4\u00e4n yritt\u00e4jyytt\u00e4 erityisesti edist\u00e4m\u00e4ll\u00e4 menetelmi\u00e4, joiden avulla my\u00f6nnet\u00e4\u00e4n kaupallisia luottoja ja liikeluottoja pienille ja keskisuurille yrityksille sek\u00e4 mikroluottoja, joiden kysynt\u00e4 on selv\u00e4sti eritt\u00e4in suuri ja joiden takaisinmaksutaso on korkea. Yleisesti ottaen haluan korostaa, ett\u00e4 sosiaalisen yhteenkuuluvuuden, vakauspolitiikan, ty\u00f6paikkojen luomisen ja ihmisten elintason tosiasiallisen parantamisen tavoitteet ovat erityisen t\u00e4rkeit\u00e4. Vaikka EBRD on pankkilaitos, sill\u00e4 on er\u00e4it\u00e4 ainutlaatuisia ominaisuuksia, koska sen teht\u00e4viin kuuluu paljon muutakin kuin raha-asioiden hoitamista tai hy\u00f6dyn tavoittelua. EBRD:n on jatkettava toimintaansa laitoksena, joka pyrkii Euroopan unionin yhteisen tavoitteen mukaisesti edist\u00e4m\u00e4\u00e4n solidaarisuutta ja yhteenkuuluvuutta. PARSEP Hugues Martinin mietint\u00f6 (A5-0448/2002) PARSEP Arvoisa puhemies, pid\u00e4n esittelij\u00e4 Martinin vesiviljely\u00e4 koskevaa mietint\u00f6\u00e4 tervetulleena. Sen ansiosta voidaan tarjota turvallinen ja suhteellisen edullinen valkuaisainel\u00e4hde ja luoda ty\u00f6paikkoja Euroopan unioniin. Kuten esittelij\u00e4 tuo esiin, tutkimusta on kuitenkin kehitett\u00e4v\u00e4 kalanviljelyst\u00e4 aiheutuvien ymp\u00e4rist\u00f6ongelmien ratkaisemiseksi erityisesti j\u00e4tehuollon osalta - keskikokoinen kalanviljelylaitos voi tuottaa yht\u00e4 paljon j\u00e4tett\u00e4 kuin pieni kaupunki. Lis\u00e4ksi meid\u00e4n on tarkasteltava kalanviljelyyn liittyvi\u00e4 elintarvikehygienian\u00e4k\u00f6kohtia, erityisesti antibioottien k\u00e4ytt\u00f6\u00e4, ja tutkittava edelleen niit\u00e4 vahinkoja, joita viljelylaitoksista vesist\u00f6\u00f6n p\u00e4\u00e4sseet kalat aiheuttavat luonnonvaraisille kalakannoille. PARSEP Meid\u00e4n on my\u00f6s tarkkailtava kalanviljelylaitoksissa viljelt\u00e4vien kalojen hyvinvointia. Jos n\u00e4m\u00e4 kysymykset ratkaistaan, vesiviljely voi tarjota, kuten aikaisemmin totesin, hyv\u00e4n ravinnonl\u00e4hteen kansalaisillemme, ja sen avulla voidaan luoda uusia ty\u00f6paikkoja. PARSEP Vesiviljelyn t\u00e4m\u00e4nhetkist\u00e4 laajuutta Euroopassa voidaan havainnollistaa sill\u00e4, ett\u00e4 Euroopassa tuotetaan 1,8 miljoonaa tonnia kalaa, \u00e4yri\u00e4isi\u00e4 ja nilvi\u00e4isi\u00e4, ja sill\u00e4, ett\u00e4 se tarjoaa 57 000 ty\u00f6paikkaa. PARSEP Kuten esittelij\u00e4 ehdottaa, meid\u00e4n on alan tulevaisuuden takaamiseksi otettava huomioon seuraavanlaisia tekij\u00f6it\u00e4: rannikko- ja samalla manneralueita koskevien yhdennettyjen ymp\u00e4rist\u00f6n k\u00e4ytt\u00f6- ja hoitosuunnitelmien laatiminen ja hallinnointi, elintarviketurvallisuuden parantaminen el\u00e4inten terveyden ja kuluttajien luottamuksen takaamiseksi, teknisen sek\u00e4 ymp\u00e4rist\u00f6\u00e4 ja terveytt\u00e4 koskevan tutkimuksen edist\u00e4minen uusien ratkaisujen l\u00f6yt\u00e4miseksi ja alan riitt\u00e4v\u00e4 rahoitus sek\u00e4 uusille yksik\u00f6ille ett\u00e4 nykyisten yksik\u00f6iden nykyaikaistamiseksi. PARSEP Edell\u00e4 mainituista syist\u00e4 ja monien luonnonvaraisten kalalajien vaikean tilanteen vuoksi vesiviljely\u00e4 voidaan pit\u00e4\u00e4 kalastusta t\u00e4ydent\u00e4v\u00e4n\u00e4 ja tulevaisuudenn\u00e4kymilt\u00e4\u00e4n varteenotettavana vaihtoehtona. PARSEP Luottamuksemme vesiviljelytuotteisiin on kuitenkin palautettava edist\u00e4m\u00e4ll\u00e4 elintarvikkeiden turvallisuutta parantavia toimenpiteit\u00e4 ja nykyaikaistamalla tuotantomenetelmi\u00e4. N\u00e4m\u00e4 seikat huomioon ottaen tuen Martinin mietint\u00f6\u00e4. PARSEP - (NL) Merikalojen viljely tuntui menneisyydess\u00e4 j\u00e4rjett\u00f6m\u00e4lt\u00e4 toimenpiteelt\u00e4. Ajateltiin, ett\u00e4 sen olivat kehitt\u00e4neet sellaiset yritykset, jotka etsiv\u00e4t uusia tilaisuuksia hy\u00f6dynt\u00e4\u00e4 suuria voittoja tuottavia markkinoita. Kun maailman v\u00e4kiluku kasvaa edelleen nopeasti ja kun meret tyhjenev\u00e4t kaloista h\u00e4lytt\u00e4v\u00e4\u00e4 vauhtia, vaikuttaa silt\u00e4, ett\u00e4 merikalojen viljelyst\u00e4 tulee v\u00e4ltt\u00e4m\u00e4t\u00f6nt\u00e4. Siihen ei ryhdyt\u00e4 uusien markkinoiden l\u00f6yt\u00e4miseksi, vaan sen vuoksi, ett\u00e4 se on olennainen osa elintarviketuotantoa. Rannikkoseutujen kalastajakyl\u00e4t, jotka ovat menett\u00e4neet vanhan tulonl\u00e4hteens\u00e4, ovat luultavasti iloisia t\u00e4st\u00e4 uudesta mahdollisuudesta. Muutokseen liittyy kuitenkin riskej\u00e4. Jos ryhdymme t\u00e4h\u00e4n hankkeeseen, tarvitsemme takuita onnettomuuksien ja ep\u00e4toivottavien tilanteiden est\u00e4miseksi. Ajatus sairaista ja ep\u00e4muodostuneista kaloista rannikkoalueiden t\u00e4p\u00f6t\u00e4ysiss\u00e4 h\u00e4keiss\u00e4, toisin sanoen uudentyyppinen bioteollisuus, on vastenmielinen. H\u00e4keiss\u00e4 el\u00e4v\u00e4t muuntogeeniset kalat voivat p\u00e4\u00e4st\u00e4 luontoon ja muuttaa peruuttamattomasti kaiken meriel\u00e4m\u00e4n esimerkiksi h\u00e4vitt\u00e4m\u00e4ll\u00e4 kaikki j\u00e4ljell\u00e4 olevat luonnonvaraiset kalat. My\u00f6s sellaisten aineiden k\u00e4ytt\u00e4minen, joita ei ole k\u00e4ytetty koskaan aikaisemmin, kalojen kasvun nopeuttamiseksi ja siten saaliin suurentamiseksi, saattaa osaltaan pilata meri\u00e4. T\u00e4m\u00e4n vuoksi on hyv\u00e4, ett\u00e4 kiinnit\u00e4mme enemm\u00e4n huomiota vesiviljelyn vaikutuksia koskeviin tietoihin, meriymp\u00e4rist\u00f6n suojelemiseen ja el\u00e4inten hyvinvointiin, jota t\u00e4m\u00e4 teollisuudenala uhkaa. PARSEP - (DA) \u00c4\u00e4nestimme kalatalousvaliokunnan kokonaismietinn\u00f6st\u00e4 tyhj\u00e4\u00e4, koska vaikka valiokunnan mietint\u00f6 on parempi kuin komission ehdotukset, valiokunnan mietinn\u00f6ss\u00e4 kannatetaan mielest\u00e4mme yleisesti ottaen EU:n vesiviljelyalan kehitt\u00e4mist\u00e4 edelleen. Koska alaan liittyy edelleen monia ongelmia, muun muassa ymp\u00e4rist\u00f6n pilaantuminen, \u00e4\u00e4nest\u00e4mme tyhj\u00e4\u00e4, vaikka tunnustammekin, ett\u00e4 valiokunta on tehnyt erinomaista ty\u00f6t\u00e4. PARSEP \u00c4\u00e4nestykset ovat p\u00e4\u00e4ttyneet. PARSEP (Istunto keskeytettiin klo 12.45 ja sit\u00e4 jatkettiin klo 15.00.) PARSEP Esityslistalla on seuraavana Migu\u00e9lez Ramosin laatima kalatalousvaliokunnan mietint\u00f6 (A5-0446/2002) kalastuksesta kansainv\u00e4lisill\u00e4 vesill\u00e4 osana yhteisen kalastuspolitiikan ulkoisia toimia (2002/2024(INI)). PARSEP Arvoisa puhemies, esittelij\u00e4 Migu\u00e9lez Ramos viittasi komission viime vuoden joulukuun 23. p\u00e4iv\u00e4n\u00e4 antamaan tiedonantoon kolmansien maiden kanssa teht\u00e4vi\u00e4 kalastussopimuksia koskevasta yhdennetyst\u00e4 toimintakehyksest\u00e4. Yhdyn h\u00e4nen n\u00e4kemykseens\u00e4 siit\u00e4, ett\u00e4 t\u00e4m\u00e4n onnekkaan yhteensattuman ansiosta parlamentin on harkittava komission ehdotuksia tarkemmin. PARSEP Tiedonannossa tarkastellaan kaikkia komission ja kolmansien maiden v\u00e4listen suhteiden kahdenv\u00e4lisyyskysymyksi\u00e4 silt\u00e4 osin kuin ne liittyv\u00e4t kalastukseen, erityisesti rannikolla sijaitsevien kehitysmaiden osalta. Tiedonanto on mielest\u00e4mme hyv\u00e4 perusta aihetta koskevalle keskustelulle. PARSEP Olemme t\u00e4ysin tietoisia siit\u00e4, kuinka t\u00e4rke\u00e4 yhteisen kalastuspolitiikan t\u00e4m\u00e4 osa on erityisesti tietyille Euroopan unionin alueille. T\u00e4m\u00e4n vuoksi komissio ehdottaa, ett\u00e4 neuvoston olisi laadittava asianmukaiset poliittiset suuntaviivat, joissa tuodaan esiin yhteis\u00f6n pyrkimys edist\u00e4\u00e4 vastuullista kalastusta ottamalla huomioon kaikkien asianomaisten edut. PARSEP Kuten esittelij\u00e4 Migu\u00e9lez Ramos huomauttaa mietinn\u00f6ss\u00e4\u00e4n varsin asianmukaisesti, yhteisen kalastuspolitiikan ulkoisen ulottuvuuden kehitt\u00e4minen on otettava huomioon Euroopan unionin kansainv\u00e4lisiss\u00e4 sitoumuksissa. Osa niist\u00e4 kuuluu YUTP:n alaan, ja osa liittyy kauppasopimuksiin. Rohkenisin sis\u00e4llytt\u00e4\u00e4 t\u00e4h\u00e4n my\u00f6s kehitysmaiden kanssa teht\u00e4v\u00e4t kest\u00e4v\u00e4\u00e4 kehityst\u00e4 koskevat sopimukset. Jatkuvia pyrkimyksi\u00e4mme parantaa politiikan ja talouden maailmanlaajuista hallintaa ei pid\u00e4 my\u00f6sk\u00e4\u00e4n sivuuttaa. PARSEP Tulevaisuuden osalta komissio katsoo, ett\u00e4 sen toimintaa olisi pidett\u00e4v\u00e4 vastuulliseen kalastukseen teht\u00e4v\u00e4n\u00e4 investointina, ei ainoastaan korvausj\u00e4rjestelyn\u00e4. PARSEP Monenv\u00e4lisyyden osalta olen pannut merkille, ett\u00e4 parlamentissa on esitetty huomautuksia henkil\u00f6resurssien ja aineellisten voimavarojen puuttumisesta. En kuitenkaan yhdy esittelij\u00e4n n\u00e4kemykseen siit\u00e4, ett\u00e4 alueellisten kalastusj\u00e4rjest\u00f6jen \u00e4\u00e4nestysj\u00e4rjestelyj\u00e4 olisi tarkistettava. Euroopan unionin vaikutusvalta n\u00e4iss\u00e4 j\u00e4rjest\u00f6iss\u00e4 on paljon suurempi kuin sen k\u00e4ytett\u00e4viss\u00e4 olevan \u00e4\u00e4nim\u00e4\u00e4r\u00e4n perusteella voitaisiin p\u00e4\u00e4tell\u00e4. PARSEP Sanktioiden ja alueellisten kalastusj\u00e4rjest\u00f6jen osalta tarvitaan mielest\u00e4mme kuitenkin enemm\u00e4n valvontaa, jotta voidaan varmistaa, ettei kansainv\u00e4lisi\u00e4 m\u00e4\u00e4r\u00e4yksi\u00e4 rikkovien alusten kalastustuotteilla voida k\u00e4yd\u00e4 kauppaa Euroopan unionissa. PARSEP Komissio on esitt\u00e4nyt \u00e4skett\u00e4in toimintasuunnitelman laittoman, ilmoittamattoman ja s\u00e4\u00e4ntelem\u00e4tt\u00f6m\u00e4n kalastuksen tukahduttamiseksi. Ehdotettujen aloitteiden, erityisesti toimenpiteiden 5-10 avulla, pyrit\u00e4\u00e4n yhdenmukaistamaan alueellisten kalastusj\u00e4rjest\u00f6jen valvontamekanismeja. Lis\u00e4ksi aloitteiden avulla j\u00e4rjestelmist\u00e4 pyrit\u00e4\u00e4n tekem\u00e4\u00e4n tehokkaampia ja taataan samalla, ett\u00e4 ne ovat monenv\u00e4lisen l\u00e4hestymistavan periaatteen mukaisia. PARSEP Minua ilahdutti se, ett\u00e4 parlamentti suhtautui suunnitelmaan l\u00e4mpim\u00e4sti. Lis\u00e4ksi t\u00e4t\u00e4 l\u00e4hestymistapaa tukeva Buskin mietint\u00f6 hyv\u00e4ksyttiin yksimielisesti viime marraskuun istunnossa. PARSEP Euroopan unioni on sitoutunut noudattamaan kaikkia kansainv\u00e4lisi\u00e4 sopimuksia, joissa se on osallisena, ja ratifioimaan New Yorkissa aikaansaadun sopimuksen. T\u00e4ll\u00e4 hetkell\u00e4 14 j\u00e4senvaltiota aikoo ratifioida kyseisen sopimuksen. Toivon, ett\u00e4 kaikki on valmiina ratifiointia varten, jotta asiassa voidaan edet\u00e4 ennen kuin n\u00e4m\u00e4 kuusi kuukautta ovat kuluneet. PARSEP Arvoisa puhemies, kuten ehk\u00e4 tied\u00e4tte, olemme tarkastelleet my\u00f6s V\u00e4limeri-kysymyst\u00e4. Toimiamme selostetaan tiedonannossa. Aiomme j\u00e4rjest\u00e4\u00e4 aihetta k\u00e4sittelev\u00e4n kansainv\u00e4lisen konferenssin Italian tulevalla puheenjohtajakaudella. Konferenssi on tarkoitus j\u00e4rjest\u00e4\u00e4 Venetsiassa joulukuussa 2003. Komissiota huolestuttaa my\u00f6s yhteis\u00f6n lains\u00e4\u00e4d\u00e4nn\u00f6n ja alueellisten kalastusj\u00e4rjest\u00f6jen suositusten t\u00e4yt\u00e4nt\u00f6\u00f6npanoa varten varatun ajan lyhent\u00e4minen. Komissio ei ole vastuussa siit\u00e4 ik\u00e4v\u00e4st\u00e4 asiasta, ett\u00e4 t\u00e4m\u00e4n ehdotuksen osalta ei ole pystytty edistym\u00e4\u00e4n. PARSEP Sosiaaliset n\u00e4k\u00f6kohdat otetaan huomioon komission ehdottamissa tavoitteissa. Niit\u00e4 on tarkoitus painottaa tulevissa yhteisty\u00f6hankkeissa. Tieteellisen tutkimuksen lis\u00e4\u00e4mistarpeen osalta toistan, ett\u00e4 komissio on esitt\u00e4nyt toiveen aihetta koskevan toimintasuunnitelman t\u00e4yt\u00e4nt\u00f6\u00f6n panemiseksi. Vaikka t\u00e4yt\u00e4nt\u00f6\u00f6npano on viiv\u00e4stynyt jonkin verran, se voidaan todenn\u00e4k\u00f6isesti saattaa menestyksekk\u00e4\u00e4sti p\u00e4\u00e4t\u00f6kseen vuoden loppuun menness\u00e4. PARSEP Lopuksi mainittakoon yhteisyritysten perustamisen edist\u00e4minen. Neuvosto on s\u00e4ilytt\u00e4nyt mahdollisuuden my\u00f6nt\u00e4\u00e4 taloudellista tukea t\u00e4llaisten yritysten perustamiselle vuoteen 2004 asti kalatalouden ohjauksen rahoitusv\u00e4lineiden (KOR) avulla. Tuki on k\u00e4ytett\u00e4viss\u00e4 kuitenkin ainoastaan, jos kalastussopimuksella voidaan taata johdonmukaiset j\u00e4rjestelyt. PARSEP Haluan lis\u00e4ksi kiinnitt\u00e4\u00e4 huomionne siihen, ettei kalastusalan assosiaatiosopimuksia koskevassa komission tiedonannossa suljeta pois t\u00e4llaisen v\u00e4lineen k\u00e4ytt\u00f6\u00e4 tulevien assosiaatiosopimusten edist\u00e4miseksi. Me p\u00e4invastoin suositamme sen k\u00e4ytt\u00f6\u00e4 yhdess\u00e4 muiden v\u00e4lineiden kanssa tulevien assosiaatioiden kehitt\u00e4misen edist\u00e4miseksi. PARSEP Luotan siihen, ett\u00e4 kalastusalan assosiaatiosopimuksia koskeva komission tiedonanto on perusta, jonka ansiosta voimme edisty\u00e4 merkitt\u00e4v\u00e4sti ja uraauurtavasti yhteisen kalastuspolitiikan ulkoisen ulottuvuuden alalla. Luotan siihen, ett\u00e4 pystymme yhdess\u00e4 luomaan motivaation sellaisen kest\u00e4v\u00e4n syv\u00e4nmerenkalastuksen takaamiseksi, jossa kaikkien edut otetaan huomioon. PARSEP Koska yhteis\u00f6n tasolla ei toteuteta mink\u00e4\u00e4nlaisia poliittisia aloitteita, komissio on vakuuttunut siit\u00e4, ett\u00e4 syv\u00e4nmeren kalastukseen k\u00e4ytett\u00e4v\u00e4n laivaston koko pienenee v\u00e4hitellen. T\u00e4m\u00e4 tuskin johtaa alusten tuhoutumiseen vaan pikemminkin siihen, ett\u00e4 aluksia siirtyy muiden lippujen alaisuuteen, tai yksityisten kalastussopimusten m\u00e4\u00e4r\u00e4n lis\u00e4\u00e4ntymiseen. PARSEP Luotan siihen, ett\u00e4 parlamentti hyv\u00e4ksyy t\u00e4m\u00e4n tilannearvion. Toivon todella, ett\u00e4 parlamentti hyv\u00e4ksyy t\u00e4m\u00e4n seurauksena sen, ett\u00e4 on t\u00e4rke\u00e4\u00e4 v\u00e4ltt\u00e4\u00e4 luomasta tilannetta, jossa yhteis\u00f6n politiikan rooli yhteis\u00f6n ulkopuolisessa kalastuksessa heikkenee merkitt\u00e4v\u00e4sti. Se olisi vastoin toivettamme luoda politiikka, jonka avulla pyrit\u00e4\u00e4n edist\u00e4m\u00e4\u00e4n kest\u00e4v\u00e4\u00e4 kalastusta kaikkialla maailmassa. PARSEP Arvoisa puhemies, kiit\u00e4n komission j\u00e4sent\u00e4 ja onnittelen esittelij\u00e4\u00e4 Migu\u00e9lez Ramosia h\u00e4nen erinomaisesta mietinn\u00f6st\u00e4\u00e4n. Kuten tied\u00e4tte, Migu\u00e9lez Ramos on kalatalousvaliokunnan ensimm\u00e4inen varapuheenjohtaja, ja h\u00e4n on er\u00e4s kyseisen valiokunnan uskollisimmista ja ahkerimmista j\u00e4senist\u00e4. Mielest\u00e4ni kalatalousvaliokunnan kaikkia espanjalaisia j\u00e4seni\u00e4 voidaan pit\u00e4\u00e4 asialle uskoutuneina ja ahkerina; t\u00e4m\u00e4 ei ehk\u00e4 ole h\u00e4mm\u00e4stytt\u00e4v\u00e4\u00e4, koska kalastusteollisuus on hyvin t\u00e4rke\u00e4 ala Espanjalle. Lis\u00e4ksi mietinn\u00f6n aihe on hyvin t\u00e4rke\u00e4 erityisesti Espanjan laivastolle, mink\u00e4 vuoksi mietinn\u00f6n laatiminen annettiin hyvin perustellusti Migu\u00e9lez Ramosin teht\u00e4v\u00e4ksi. PARSEP Mielest\u00e4ni on t\u00e4rke\u00e4\u00e4 panna merkille, ett\u00e4 Espanjan laivastolla on pitk\u00e4t perinteet kaukaisilla vesill\u00e4 kalastamisessa. Espanjalaiset kalastajat ovat kalastaneet Newfoundlandin ja Labradorin rannikkovesill\u00e4 yli tuhat vuotta - satoja vuosia ennen ajankohtaa, jona Christopher Kolumbuksen v\u00e4itet\u00e4\u00e4n l\u00f6yt\u00e4neen Amerikan - ja pid\u00e4mme nyt juuri t\u00e4t\u00e4 kaukaisten vesien laivastoa t\u00e4rke\u00e4n\u00e4 osana yhteis\u00f6n yleist\u00e4 kalastusstrategiaa. PARSEP T\u00e4m\u00e4naamuisessa keskustelussamme monet puhujat viittasivat siihen, ett\u00e4 vesiviljelyteollisuudella on entist\u00e4kin t\u00e4rke\u00e4mpi asema kuluttajien kalatuotteiden kysynt\u00e4\u00e4n vastaamisessa, koska kalakannat ovat romahtaneet EU:n vesill\u00e4. Eurooppalaisten kalastajien osallistuminen kalastukseen kansainv\u00e4lisill\u00e4 vesill\u00e4 on yht\u00e4 t\u00e4rke\u00e4\u00e4. Yhteis\u00f6n omavaraisuusaste on ainoastaan noin 60 prosenttia kalojen kulutuksen osalta, ja kalatuotteiden kysynt\u00e4 on kasvussa BSE:n ja suu- ja sorkkataudin vuoksi ja kuluttajien v\u00e4hent\u00e4ess\u00e4 asteittain punaisen lihan k\u00e4ytt\u00f6\u00e4. Kansainv\u00e4lisill\u00e4 vesill\u00e4 harjoitettavan kalastuksen rooli on eritt\u00e4in t\u00e4rke\u00e4 kuluttajien kysynt\u00e4\u00e4n vastaamiseksi ja mahdollisimman monien kalastusalan ty\u00f6paikkojemme s\u00e4ilytt\u00e4miseksi. Migu\u00e9lez Ramos on tuonut mietinn\u00f6ss\u00e4\u00e4n ja alkuhuomautuksissaan selke\u00e4sti esiin, ett\u00e4 jokainen yhteis\u00f6n kansainv\u00e4lisill\u00e4 vesill\u00e4 tapahtuvaan kalastukseen sijoittama euro synnytt\u00e4\u00e4 kolmen euron edest\u00e4 liiketoimintaa. PARSEP Sen lis\u00e4ksi, ett\u00e4 sijoittaminen kalastukseen on hyv\u00e4 keino alan ty\u00f6paikkojen s\u00e4ilytt\u00e4miseksi, se on yhteis\u00f6n kannalta hyv\u00e4\u00e4 ja taloudellisesti kannattavaa liiketoimintaa. Mielest\u00e4ni t\u00e4ss\u00e4 yhteydess\u00e4 on syyt\u00e4 mainita my\u00f6s, ett\u00e4 EU:n ulkopuolisten maiden kanssa tehtyj\u00e4 sopimuksia (sopimukset kolmansien maiden kanssa) koskeva yleinen mielipide on ollut kielteinen, ja joissain tapauksissa kielteinen mielipide on saattanut olla aiheellinen. Juuri t\u00e4m\u00e4n vuoksi Migu\u00e9lez Ramos vaatii mietinn\u00f6ss\u00e4\u00e4n kansainv\u00e4lisill\u00e4 vesill\u00e4 harjoitettavan kalastuksen alan tarkastusten, valvonnan ja j\u00e4rjestyksenpidon sek\u00e4 erityisesti laittoman kalastuksen torjunnan tehostamista. Takaamalla t\u00e4llaisten sopimusten asianmukaisen valvonnan ja kohdentamalla merkitt\u00e4v\u00e4n rahoitusosuuden suojeluun ja kehitt\u00e4miseen pystymme vastaamaan aiempaa paremmin arvostelijoidemme varsin perusteltuihin huolenaiheisiin. PARSEP Suositan mietint\u00f6\u00e4 parlamentille. PARSEP Arvoisa puhemies, kalastus kansainv\u00e4lisill\u00e4 vesill\u00e4 edellytt\u00e4\u00e4 vastuullisuutta, eik\u00e4 p\u00e4\u00e4m\u00e4\u00e4r\u00e4n\u00e4mme saa olla hy\u00f6dyntavoittelu. Emme voi jatkuvasti siirt\u00e4\u00e4 ongelmiamme muualle. PARSEP Toivon, ett\u00e4 parlamentti tukee parlamentin j\u00e4senen Attwoollin tarkistuksia, jotka parantavat mietint\u00f6\u00e4 jonkin verran. Ryhm\u00e4ni \u00e4\u00e4nest\u00e4\u00e4 ehdottomasti n\u00e4iden tarkistusten puolesta. Emme kuitenkaan \u00e4\u00e4nest\u00e4 mietinn\u00f6n puolesta edes siin\u00e4 tapauksessa, ett\u00e4 - kuten toivomme - kyseiset tarkistukset hyv\u00e4ksyt\u00e4\u00e4n, koska se perustuu n\u00e4kemykselle, jonka mukaan Euroopan unioni voi vied\u00e4 ylikapasiteettiaan muualle ja jonka mukaan meill\u00e4 on joistain historiallisista syist\u00e4 oikeus riist\u00e4\u00e4 kehitysmaita. PARSEP On totta, ett\u00e4 monet j\u00e4senvaltiot ovat pit\u00e4neet muita maita siirtomainaan ja k\u00e4ytt\u00e4neet heikompia ja k\u00f6yhempi\u00e4 maita hyv\u00e4kseen ep\u00e4oikeudenmukaisesti vuosisatojen ajan, mutta meid\u00e4n 2000-lukua el\u00e4vien olisi ymm\u00e4rrett\u00e4v\u00e4, ett\u00e4 imperialistinen l\u00e4hestymistapa on v\u00e4\u00e4r\u00e4. Kehitysmaihin kohdistuvien sosiaalisten ja taloudellisten vaikutusten tarkastelu osoittaa, ett\u00e4 kalastussopimuksissa on paljon toivomisen varaa. PARSEP T\u00e4m\u00e4 ei ole yksinomaan Espanjan ongelma. H\u00e4pe\u00e4kseni minun on todettava, ett\u00e4 oman maani Irlannin Atlantic Dawn -alus on my\u00f6s er\u00e4s esimerkki riistosta. Monien muiden suuralusten tavoin se toimii kehitysmaissa ik\u00e4\u00e4n kuin sill\u00e4 olisi oikeus hy\u00f6dynt\u00e4\u00e4 muiden maiden kalavaroja ilman mink\u00e4\u00e4nlaista valvontaa. Se on itse asiassa muita pahempi, koska se on er\u00e4s kaikkein suurimmista t\u00e4llaisista aluksista. Huolimatta Euroopan unionin sopimusten osalta esitetyist\u00e4 toteamuksista valvontaa harjoitetaan sent\u00e4\u00e4n jossain m\u00e4\u00e4rin, vaikkakin sopimukset j\u00e4tt\u00e4v\u00e4t paljon toivomisen varaa. Yksityisten sopimusten yhteydess\u00e4 ei kuitenkaan harjoiteta mink\u00e4\u00e4nlaista valvontaa. PARSEP Meid\u00e4n on tarkasteltava asennettamme kehitysmaihin. Tuen j\u00e4sen Attwoollin esitt\u00e4mi\u00e4 n\u00e4kemyksi\u00e4. Toimintamme on oltava sopusoinnussa puheidemme kanssa. PARSEP Arvoisa puhemies, t\u00e4m\u00e4 mietint\u00f6 annetaan Euroopan parlamentin k\u00e4sitelt\u00e4v\u00e4ksi ajankohtana, jona yleinen huolestuneisuus Euroopan unionin sopimusten vaikutuksista kolmansiin maihin kasvaa yh\u00e4. Erityisesti Maailman Luonnons\u00e4\u00e4ti\u00f6 on tuominnut kyseiset sopimukset hyvin \u00e4\u00e4nekk\u00e4\u00e4sti. Mielest\u00e4ni on rohkaisevaa, ett\u00e4 jopa t\u00e4\u00e4ll\u00e4 parlamentissa yli 100 j\u00e4sent\u00e4 \u00e4\u00e4nesti Senegalin sopimusta vastaan kuukausi sitten. H\u00e4mm\u00e4styksekseni olen kerrankin samaa mielt\u00e4 jopa oman maani ulkoministeri\u00f6n kanssa. Jopa se my\u00f6nt\u00e4\u00e4, etteiv\u00e4t n\u00e4m\u00e4 sopimukset ole toimivia. PARSEP Mietinn\u00f6ss\u00e4 on hyvin puolustava s\u00e4vy. Siin\u00e4 todetaan, ett\u00e4 EU:n kalastajat ?eiv\u00e4t ole merirosvoja?. S\u00e4\u00e4nt\u00f6j\u00e4 on kuitenkin rikottu niin paljon, ett\u00e4 t\u00e4m\u00e4 v\u00e4ite on selv\u00e4sti ristiriidassa tilintarkastustuomioistuimen t\u00e4m\u00e4nhetkisen kertomuksen kanssa. H\u00e4iritsevint\u00e4 on esittelij\u00e4n ja komission j\u00e4senen taka-ajatus siit\u00e4, ett\u00e4 jos emme k\u00e4yt\u00e4 n\u00e4it\u00e4 kalastusoikeuksia, jokin muu kansa k\u00e4ytt\u00e4\u00e4 niit\u00e4 ja ett\u00e4 t\u00e4m\u00e4 saattaisi olla jollakin tavoin viel\u00e4 pahempi asia. Vaikka t\u00e4m\u00e4 pit\u00e4isi paikkansa, ei voida katsoa, ett\u00e4 kaksi v\u00e4\u00e4r\u00e4\u00e4 johtaa oikeaan lopputulokseen - enk\u00e4 uskoisi, ett\u00e4 n\u00e4in lapsellisia v\u00e4itteit\u00e4 esitet\u00e4\u00e4n. Palauttakaamme mieliin, mit\u00e4 n\u00e4m\u00e4 ?v\u00e4\u00e4r\u00e4t? ovat. PARSEP Sopimukset aiheuttavat ymp\u00e4rist\u00f6tuhoja v\u00e4hemm\u00e4n kehittyneille maille, mik\u00e4 on biologisesti kuin Serengetin sytytt\u00e4minen tuleen. T\u00e4m\u00e4 on er\u00e4s niist\u00e4 imperialismin muodoista, joihin Euroopan unioni syyllistyy: toimimalla n\u00e4in se est\u00e4\u00e4 n\u00e4it\u00e4 kehitysmaita, jotka voisivat muuten parantaa omaa talouttaan, kehittym\u00e4st\u00e4. Lis\u00e4ksi n\u00e4m\u00e4 sopimukset ovat moraalisesti vastenmielisi\u00e4, koska ne aiheuttavat vuosittain suoraan ja v\u00e4lillisesti satojen mustan Afrikan alkuper\u00e4isv\u00e4est\u00f6\u00f6n kuuluvien kalastajien kuoleman. PARSEP Mietinn\u00f6ss\u00e4 meit\u00e4 pyydet\u00e4\u00e4n sulkemaan silm\u00e4mme, vaikka itse asiassa ihmiset ovat alkaneet yh\u00e4 kasvavassa m\u00e4\u00e4rin avata silm\u00e4ns\u00e4. Meid\u00e4n olisi t\u00e4m\u00e4n vuoksi hyl\u00e4tt\u00e4v\u00e4 mietint\u00f6. Ep\u00e4ilen, ett\u00e4 enemmist\u00f6 kuitenkin kannattaa mietint\u00f6\u00e4, mutta varoitan, ett\u00e4 ihmiset ovat havahtumassa, eik\u00e4 t\u00e4llaisia sopimuksia hyv\u00e4ksyt\u00e4 en\u00e4\u00e4 tulevaisuudessa. PARSEP Arvoisa puhemies, hyv\u00e4 komission j\u00e4sen, parlamentissa t\u00e4n\u00e4\u00e4n osoittamanne voimakas kiinnostus - vaikka ette olekaan itse vastuussa kalastuksesta - on toivoakseni selvent\u00e4nyt sit\u00e4, kuinka paljon tukea ajatuksemme saavat sek\u00e4 kalatalousvaliokunnassa ett\u00e4 komissiossa. PARSEP Olemme t\u00e4n\u00e4\u00e4n tuoneet selv\u00e4sti esiin, ett\u00e4 emme halua harjoittaa koeputkikalastusta. Emme halua ainoastaan vesiviljeltyj\u00e4 kaloja tai vakiopakkauksia valintamyym\u00e4l\u00e4n hyllylt\u00e4. Haluamme enemm\u00e4n, vaikka meremme voivatkin huonosti. PARSEP Sy\u00f6m\u00e4mme kala on yh\u00e4 harvemmin per\u00e4isin Euroopan unionin vesilt\u00e4. Oman maanosamme tarjonta ei vastaa Euroopan kysynt\u00e4\u00e4. Oman maanosamme tarjonnan tulevaisuudenn\u00e4kym\u00e4t eiv\u00e4t ole riitt\u00e4v\u00e4t. T\u00e4m\u00e4n vuoksi tarvitsemme kalastusta ulkomaisilla vesill\u00e4, sek\u00e4 kansainv\u00e4lisill\u00e4 vesill\u00e4 ett\u00e4 muiden valtioiden rannikkoalueilla. Esittelij\u00e4 Migu\u00e9lez Ramos, kiit\u00e4n teit\u00e4 siit\u00e4, ett\u00e4 olette tarkastellut t\u00e4t\u00e4 kysymyst\u00e4 yksityiskohtaisessa valiokunta-aloitteisessa mietinn\u00f6ss\u00e4 ja m\u00e4\u00e4ritellyt t\u00e4rkeimm\u00e4t ongelmat t\u00e4sm\u00e4llisesti. PARSEP Euroopan unionin kalastussopimukset ovat t\u00e4rke\u00e4 tekij\u00e4 sek\u00e4 tarjonnan ett\u00e4 ty\u00f6llisyyden kannalta, etenkin kalastuksesta riippuvaisilla alueilla. Mietinn\u00f6n hylk\u00e4\u00e4minen, mit\u00e4 juuri vaadittiin, ei voi olla tavoitteemme, sill\u00e4 meid\u00e4n on tarkasteltava n\u00e4it\u00e4 kysymyksi\u00e4 vakavasti kaikista mahdollisista ristiriidoista huolimatta. PARSEP Emme saa my\u00f6sk\u00e4\u00e4n unohtaa sit\u00e4, ett\u00e4 olemme hy\u00f6dynt\u00e4neet omia vesi\u00e4mme niin paljon, ett\u00e4 se on johtanut kalakantojen ehtymiseen. Meid\u00e4n on vain harkittava yhteisen kalastuspolitiikan uudistamista. Kuten er\u00e4s edellisist\u00e4 puhujista totesi, olisi kohtalokasta pelk\u00e4st\u00e4\u00e4n vied\u00e4 n\u00e4m\u00e4 ongelmat muualle. PARSEP Vaadin kest\u00e4v\u00e4\u00e4 l\u00e4hestymistapaa kalastussopimuksiin ja voittojen tasapuolista jakoa, mutta luonnollisesti my\u00f6s taloudellisesti kannattavaa l\u00e4hestymistapaa, sill\u00e4 en halua sivuuttaa tosiasioita. Pitk\u00e4aikaiset julkiset tuet ovat kuitenkin ristiriidassa t\u00e4m\u00e4n p\u00e4\u00e4m\u00e4\u00e4r\u00e4n kanssa. On j\u00e4rjet\u00f6nt\u00e4 perustella kalakantojen vastuutonta hy\u00f6dynt\u00e4mist\u00e4 muilla vesill\u00e4 laivaston koon pienentymisell\u00e4 ja Euroopan unionin markkinoiden tarjonnalla. PARSEP Lopuksi todettakoon, ett\u00e4 t\u00e4rkeimp\u00e4n\u00e4 p\u00e4\u00e4m\u00e4\u00e4r\u00e4n\u00e4mme on oltava pysyvien kalavarojen suojeleminen sek\u00e4 Euroopassa ett\u00e4 ulkomaisilla vesill\u00e4. T\u00e4lt\u00e4 osin meid\u00e4n on kuitenkin hyv\u00e4ksytt\u00e4v\u00e4 vastuumme aidosti eik\u00e4 ainoastaan pyritt\u00e4v\u00e4 lyhytn\u00e4k\u00f6isesti miellytt\u00e4m\u00e4\u00e4n. T\u00e4m\u00e4 koskee erityisesti kansainv\u00e4lisi\u00e4 vesi\u00e4. Laittoman kalastuksen tehokas torjunta edellytt\u00e4\u00e4 m\u00e4\u00e4r\u00e4tietoisia toimia. Pelk\u00e4t tarkastukset eiv\u00e4t ole riitt\u00e4v\u00e4 toimenpide. Sanktiot on yhdenmukaistettava kansainv\u00e4lisesti. T\u00e4m\u00e4 on mahdollista ainoastaan toimimalla yhteisty\u00f6ss\u00e4 ja maailmanlaajuisen kehyksen puitteissa. Sanktiot eiv\u00e4t saa olla pelk\u00e4st\u00e4\u00e4n neuvoa-antavia. Tied\u00e4n, ett\u00e4 kansainv\u00e4lisen yhteis\u00f6n koneistot - ja my\u00f6s omat koneistomme - toimivat hitaasti, mutta k\u00e4yt\u00e4mme niit\u00e4 edelleen ja pid\u00e4mme ne liikkeess\u00e4. PARSEP Arvoisa puhemies, yhdyn kollegani ja maanmieheni Rosa Migu\u00e9lez Ramosin mietinn\u00f6ss\u00e4\u00e4n esitt\u00e4miin n\u00e4kemyksiin. Olen h\u00e4nen kanssaan samaa mielt\u00e4 siit\u00e4, ett\u00e4 kalastus kansainv\u00e4lisill\u00e4 vesill\u00e4 on ratkaisevan t\u00e4rke\u00e4\u00e4 Euroopan unionille, kalastusalalle ja kansainv\u00e4liselle politiikalle sek\u00e4 muiden maiden kanssa teht\u00e4v\u00e4lle taloudelliselle ja sosiaaliselle yhteisty\u00f6lle. PARSEP Euroopan unionin todellinen politiikka ei ole kuitenkaan sopusoinnussa t\u00e4m\u00e4n n\u00e4kemyksen kanssa, ja sama koskee yhteis\u00f6n vesill\u00e4 harjoitettavaa kalastusta. Yhteisen kalastuspolitiikan ep\u00e4onnistunut uudistus toi esiin t\u00e4m\u00e4n ik\u00e4v\u00e4n tosiseikan. Minun on todettava t\u00e4ss\u00e4 yhteydess\u00e4, ett\u00e4 kotimaani Galicia on er\u00e4s eniten k\u00e4rsineist\u00e4 alueista; k\u00e4rsimme (...) tiettyjen maiden ja laivastojen vuoksi ja kansainv\u00e4lisill\u00e4 vesill\u00e4, ja k\u00e4rsimme komission ja neuvoston ylimielisyydest\u00e4, josta t\u00e4m\u00e4 kansainv\u00e4linen politiikka on osoituksena. Komission yksik\u00f6ill\u00e4 ei ole edes tarvittavia henkil\u00f6resursseja eik\u00e4 taloudellisia ja aineellisia voimavaroja. Jopa manterella harjoitettava kalastus ja vesiviljely k\u00e4rsiv\u00e4t j\u00e4lleen, kuten aina, vakavista vahingoista Galician rannikon - kuten Prestigest\u00e4 aiheutuneiden - tuhojen vuoksi. T\u00e4m\u00e4 tilanne on korjattava pikaisesti. PARSEP Keskustelu on p\u00e4\u00e4ttynyt. PARSEP \u00c4\u00e4nestys toimitetaan iltap\u00e4iv\u00e4n keskustelujen j\u00e4lkeen. PARSEP Esityslistalla on seuraavana komission julkilausuma matalataajuisen kaikuluotaimen vaikutuksista meren eli\u00f6st\u00f6\u00f6n. PARSEP Arvoisa puhemies, haluan komission puolesta kiitt\u00e4\u00e4 parlamenttia pyynn\u00f6st\u00e4 laatia julkilausuma t\u00e4st\u00e4 aiheesta. PARSEP Komissio on tyytyv\u00e4inen siihen, ett\u00e4 sill\u00e4 on j\u00e4lleen kerran tilaisuus korostaa, miten t\u00e4rke\u00e4n\u00e4 yhteis\u00f6 pit\u00e4\u00e4 meriymp\u00e4rist\u00f6n ja sen monimuotoisuuden suojelua. PARSEP Parlamentti ja neuvosto ovat jo korostaneet aiheen t\u00e4rkeytt\u00e4 kuudennessa ymp\u00e4rist\u00f6\u00e4 koskevassa yhteis\u00f6n toimintaohjelmassa. Yksi t\u00e4m\u00e4n ohjelman tavoitteista on varmistaa, ettei luonnon monimuotoisuus v\u00e4hene en\u00e4\u00e4 vuoden 2010 j\u00e4lkeen. PARSEP Luonnonvaraisen el\u00e4imist\u00f6n ja kasviston suojelu ovat my\u00f6s Euroopan komission meriymp\u00e4rist\u00f6n suojelua ja s\u00e4ilytt\u00e4mist\u00e4 koskevan strategian kaksi tavoitetta. PARSEP Valaat ovat haavoittuva osa meren eli\u00f6st\u00f6\u00e4. Ihmisten monet toimet aiheuttavat varmasti niille vaaraa, mutta aina ei ole helppoa vahvistaa n\u00e4iden toimien ja el\u00e4inten tilanteen v\u00e4list\u00e4 yhteytt\u00e4. Komissio seuraa tarkkaan keskustelua matalataajuisen kaikuluotaimen k\u00e4yt\u00f6n mahdollisista vaikutuksista joihinkin valaslajeihin. Saattaa olla, etteiv\u00e4t matalalla taajuudella l\u00e4hetetty kova \u00e4\u00e4ni h\u00e4iritse n\u00e4it\u00e4 el\u00e4imi\u00e4. Tarvitaan kuitenkin tarkempaa tutkimusta siit\u00e4, miten vakavasti l\u00e4hetykset vaikuttavat niihin. T\u00e4st\u00e4 huolimatta nykyist\u00e4 ep\u00e4tietoisuutta l\u00e4hetysten vaikutuksista ei voida k\u00e4ytt\u00e4\u00e4 tekosyyn\u00e4 sille, ettei toimenpiteit\u00e4 toteuteta asian k\u00e4sittelemiseksi. PARSEP Komissio katsoo, ett\u00e4 kyseisen tekniikan k\u00e4yt\u00f6ss\u00e4 tulisi noudattaa erityist\u00e4 varovaisuutta, kun otetaan huomioon mahdolliset vaikutukset meriymp\u00e4rist\u00f6\u00f6n. Ihanteellista olisi, ett\u00e4 mahdolliset vaikutukset arvioitaisiin etuk\u00e4teen. PARSEP Haluan palauttaa mieliinne, ett\u00e4 valaita suojellaan luontotyyppidirektiivin 92/43/ETY nojalla. Direktiiviss\u00e4 j\u00e4senvaltiot velvoitetaan muun muassa toteuttamaan tarvittavia toimenpiteit\u00e4 valaiden h\u00e4iritsemisen est\u00e4miseksi. PARSEP N\u00e4ist\u00e4 syist\u00e4 ja Kanariansaarten \u00e4skett\u00e4isten tapahtumien takia komissio pyysi luontotyyppikomitean 20. marraskuuta pidetyss\u00e4 kokouksessa lis\u00e4tietoja aiheesta sellaisten j\u00e4senvaltioiden kansallisilta viranomaisilta, joissa vastaavia tapauksia on sattunut. L\u00e4het\u00e4mme nyt j\u00e4senvaltioille muistutuksen sen varmistamiseksi, ett\u00e4 ne vastaavat pyynt\u00f6\u00f6mme. Komissio toimittaa kaikki saamansa vetoomukseen liittyv\u00e4t tiedot parlamentille. PARSEP Lis\u00e4ksi komissio pyyt\u00e4\u00e4 kansainv\u00e4lisen merentutkimusneuvoston kaltaisia riippumattomia tieteellisi\u00e4 elimi\u00e4 arvioimaan nykyist\u00e4 tilannetta sek\u00e4 saatavilla olevia tietoja aiheesta. On toivottavaa, ett\u00e4 matalataajuisen kaikuluotaimen mahdollisista vaikutuksista meriymp\u00e4rist\u00f6\u00f6n saataisiin enemm\u00e4n tietoa. Ainoastaan, kun luotettavaa tieteellist\u00e4 tietoa on saatavissa, on mahdollista p\u00e4\u00e4tt\u00e4\u00e4 nykyisin k\u00e4yt\u00f6ss\u00e4 olevia toimenpiteit\u00e4 t\u00e4ydent\u00e4vien tarvittavien toimenpiteiden luonteesta. PARSEP Arvoisa puhemies, haluan kiitt\u00e4\u00e4 puhemiesneuvostoa siit\u00e4, ett\u00e4 t\u00e4m\u00e4 t\u00e4rke\u00e4 asia on otettu k\u00e4sittelyyn, ja haluan kiitt\u00e4\u00e4 arvoisaa komission j\u00e4sent\u00e4. PARSEP Aivan aluksi on todettava, ett\u00e4 on t\u00e4ysin oikeutettua pyrki\u00e4 kehitt\u00e4m\u00e4\u00e4n puolustusj\u00e4rjestelm\u00e4\u00e4 valvomaan uuden teknologian sukellusveneit\u00e4, joita on vaikea havaita perinteisill\u00e4 passiivisilla tutkilla. On selv\u00e4\u00e4, ett\u00e4 tutkassa n\u00e4kym\u00e4tt\u00f6mi\u00e4 sukellusveneit\u00e4 k\u00e4ytt\u00e4v\u00e4t vain ne, joiden tarkoitusper\u00e4t ovat vihamielisi\u00e4. Siksi on syyt\u00e4 olla ajan tasalla. Kysymys kuitenkin on siit\u00e4, millaisen teknologian valitsemme ja mik\u00e4 on sen ekologinen hinta. PARSEP Nato-maiden kehittelem\u00e4 matalilla taajuuksilla toimiva tutkaj\u00e4rjestelm\u00e4 eli LFAS saattaa aiheuttaa valaiden harhautumisia rannoille sek\u00e4 kuolemaan johtavia sis\u00e4isi\u00e4 vaurioita niiden sis\u00e4elimiss\u00e4. Tutkan tuottama korkeadesibelinen ja matalataajuuksinen \u00e4\u00e4ni voi kulkea meress\u00e4 tuhansia kilometrej\u00e4. PARSEP Valaskuolemia on sattunut alueilla, joilla Yhdysvallat on testannut j\u00e4rjestelm\u00e4\u00e4. Maaliskuussa 2000 Bahamasaarilla 17 valasta ajautui rantaan ja kahdeksan niist\u00e4 kuoli samaan aikaan, kun merell\u00e4 testattiin laivaston tutkaj\u00e4rjestelm\u00e4\u00e4. Kuolleille valaille tehdyiss\u00e4 ruumiinavauksissa todettiin \u00e4\u00e4rimm\u00e4isen kovan \u00e4\u00e4nen tuottamia verenpurkaumia aivoissa ja sis\u00e4korvissa. Yhdysvaltain laivasto itse totesi raportissaan Bahamasaarten valaskuolemista LFAS:n olleen todenn\u00e4k\u00f6isin syy valaskuolemiin. PARSEP Syyskuussa 2002 Kanariansaarilla, siis EU:n alueella, tapahtui isoimpia valaskuolemia 20 vuoteen, kun 15 eri lajien valasta kuoli sis\u00e4isiin verenpurkaumiin. Nato piti samaan aikaan alueella sotaharjoitusta, johon osallistui 58 sota-alusta ja kuusi sukellusvenett\u00e4 liittoutuman eri j\u00e4senvaltioista. PARSEP LFAS-laite tuottaa 1000 hertsin matalataajuista ja keskim\u00e4\u00e4rin 250 desibelin \u00e4\u00e4nt\u00e4, jolla se haravoi merenpohjaa satojen kilometrien et\u00e4isyyksilt\u00e4. PARSEP Kovan mutta matalataajuuksisen \u00e4\u00e4nen vaikutukset valaiden k\u00e4ytt\u00e4ytymiseen ja sis\u00e4elimiin ovat moninaisia. Melu saattaa aiheuttaa korvaluiden tuhoutumisen sek\u00e4 verenvuotoja aivoissa ja keuhkoissa. J\u00e4rjestelm\u00e4 saattaa sekoittaa viel\u00e4 kaukanakin \u00e4\u00e4nil\u00e4hteest\u00e4 valaiden kommunikaatioj\u00e4rjestelm\u00e4\u00e4 toimiessaan samoilla matalilla taajuuksilla, joita valaat itsekin k\u00e4ytt\u00e4v\u00e4t. Melu vaikeuttaa valaiden suunnistusta, laumojen muodostusta, parittelua sek\u00e4 ravinnonsaantia. Tutkan ja valaskuolemien yhteytt\u00e4 on toki haastavaa ja vaikeaa todistaa, sill\u00e4 valaiden j\u00e4ljitt\u00e4minen ja niiden k\u00e4ytt\u00e4ytymisen muutosten tutkiminen pitk\u00e4ll\u00e4 aikav\u00e4lill\u00e4 on eritt\u00e4in vaikeaa. PARSEP Yhdysvaltain laivasto on tehnyt j\u00e4rjestelm\u00e4st\u00e4 ymp\u00e4rist\u00f6vaikutusten arvioinnin, mutta hyv\u00e4maineiset amerikkalaiset ymp\u00e4rist\u00f6j\u00e4rjest\u00f6t pit\u00e4v\u00e4t sit\u00e4 puutteellisena. Liittovaltion tuomari kielsi viime marraskuussa laivastoa toistaiseksi testaamasta j\u00e4rjestelm\u00e4\u00e4 Yhdysvaltain aluevesill\u00e4 vetoamalla merinis\u00e4kk\u00e4iden suojelulakiin. Laivastolla on kuitenkin kansallisen viranomaisen lupa j\u00e4rjestelm\u00e4n testaamiseen 432 tunnin ajan muilla merialueilla kuluvan vuoden elokuuhun asti. PARSEP Laivasto on t\u00e4h\u00e4n saakka suostunut testaamaan laitteistoaan et\u00e4\u00e4ll\u00e4 rannikkoalueilta sek\u00e4 poissa merinis\u00e4kk\u00e4iden tunnetuilta lis\u00e4\u00e4ntymis- tai l\u00e4pikulkualueilta. Tutkijat ovat kuitenkin todenneet, ettei testaaminen ole turvallista edes keskell\u00e4 Tyynt\u00e4 valtamerta, sill\u00e4 valaiden reittej\u00e4 on mahdotonta tiet\u00e4\u00e4 kattavasti. J\u00e4rjestelem\u00e4n avulla voidaan kattaa jopa 75 prosenttia maailman merist\u00e4. PARSEP T\u00e4ysin selvitt\u00e4m\u00e4tt\u00e4 on, miten j\u00e4rjestelm\u00e4 vaikuttaa V\u00e4limerell\u00e4 tai It\u00e4merell\u00e4, joilla tutkan kantama on rannasta rantaan eik\u00e4 el\u00e4imill\u00e4 ole mahdollisuutta siirty\u00e4 pois \u00e4\u00e4nen vaikutusalueelta. EU:lla on selv\u00e4 velvoite toimia ja vaatia tietoja j\u00e4rjestelm\u00e4st\u00e4. PARSEP Asiaan liittyy my\u00f6s selke\u00e4 taloudellinen intressi: EU:n alueella toimivat kalastajat ovat olleet viime vuoden aikana huolestuneita tutkaj\u00e4rjestelm\u00e4n vaikutuksista kalakantoihin kalastusvesill\u00e4\u00e4n. Vaikka kalat olisivatkin v\u00e4hemm\u00e4n herkki\u00e4 j\u00e4rjestelm\u00e4n haittavaikutuksille kuin merinis\u00e4kk\u00e4\u00e4t, on selv\u00e4\u00e4, ett\u00e4 tutkan melu saattaa karkottaa kaloja ja siten haitata ammatinharjoittamista. PARSEP Arvoisa puhemies, kiit\u00e4n komissiota julkilausumasta. Edell\u00e4 mainitun perusteella on pyritt\u00e4v\u00e4 saamaan aikaiseksi tutkaj\u00e4rjestelm\u00e4n k\u00e4ytt\u00f6kielto toistaiseksi ja selvitett\u00e4v\u00e4 nopeasti j\u00e4rjestelm\u00e4n vaikutukset meren eli\u00f6st\u00f6\u00f6n. Olisi parasta, ett\u00e4 t\u00e4n\u00e4\u00e4n kuullun komission julkilausuman perusteella ja muualta saatavan aineiston pohjalta Euroopan parlamentin ymp\u00e4rist\u00f6valiokunta ottaisi asian k\u00e4sittelyyn ja laatisi asiasta mietinn\u00f6n. PARSEP T\u00e4m\u00e4 kysymys aiheuttaa yh\u00e4 enemm\u00e4n huolta sek\u00e4 kansalaisille ett\u00e4 parlamentille, mit\u00e4 osoittaa selke\u00e4sti my\u00f6s se, ett\u00e4 saimme aihetta koskevaan suulliseen kysymykseemme 60 allekirjoitusta. PARSEP Kuten kollegani Korhola juuri totesi, on olemassa entist\u00e4 enemm\u00e4n n\u00e4ytt\u00f6\u00e4 siit\u00e4, ett\u00e4 eritt\u00e4in korkea\u00e4\u00e4niset kaikuluotaimet muodostavat suuren uhan merinis\u00e4kk\u00e4ille ja kaloille. Pelottavaa t\u00e4ss\u00e4 on se, ett\u00e4 t\u00e4m\u00e4 teknologia altistaa ne melusaasteelle, joka on 200 miljardia kertaa suurempi kuin melu, jolle ne nykyisin altistuvat, joten ei ole yll\u00e4tt\u00e4v\u00e4\u00e4, ett\u00e4 valaita on kuollut keuhkoverenvuodon, kuulon vahingoittumisen sek\u00e4 ravinnon saannin, parittelun ja viestinn\u00e4n vaikeutumisen takia. PARSEP Kyseinen teknologia on my\u00f6s Yhdistyneiden Kansakuntien merioikeusyleissopimuksen vastainen. Sopimuksessa valtioita vaaditaan toteuttamaan kaikki tarvittavat toimenpiteet, joilla estet\u00e4\u00e4n, v\u00e4hennet\u00e4\u00e4n ja valvotaan meriymp\u00e4rist\u00f6n pilaantumista l\u00e4hteest\u00e4 riippumatta. Nyt Naton ja sen j\u00e4senmaiden on toimittava, ja koska yli puolet Naton 19 j\u00e4senmaasta on Euroopan unionin j\u00e4senvaltioita, EU:lla on erityinen velvollisuus edist\u00e4\u00e4 toimiin. Ei riit\u00e4, ett\u00e4 komissio vastaa 29. marraskuuta 2002 esitt\u00e4m\u00e4\u00e4ni kirjalliseen kysymykseen toteamalla ainoastaan, ett\u00e4 t\u00e4m\u00e4ntyyppist\u00e4 tekniikkaa on k\u00e4ytett\u00e4v\u00e4 varoen. Tosiasia on, ettei t\u00e4m\u00e4ntyyppist\u00e4 tekniikkaa voi k\u00e4ytt\u00e4\u00e4 varoen, kun otetaan huomioon, ett\u00e4 sadan mailin et\u00e4isyydell\u00e4 melun l\u00e4hteen\u00e4 olevasta aluksesta \u00e4\u00e4ni voi yh\u00e4 aiheuttaa kudosten repeytymist\u00e4 esimerkiksi valaiden ja delfiinien aivojen takana olevassa ilmapussissa. T\u00e4m\u00e4n takia haluamme, ett\u00e4 Nato-maiden laivastoille asetetaan LFAS:n v\u00e4lit\u00f6n k\u00e4ytt\u00f6kielto ja ett\u00e4 suoritetaan perusteellinen ja riippumaton vaikutusten arviointi. Haluamme, ett\u00e4 Yhdysvallat ja Nato-maat tekev\u00e4t merioikeusyleissopimuksen 206 artiklassa vaaditun tutkimuksen LFAS:n mahdollisista haitallisista vaikutuksista ja julkaisevat sen. PARSEP Toivoimme, ett\u00e4 my\u00f6s neuvosto olisi ollut valmis antamaan julkilausuman LFAS:sta t\u00e4\u00e4ll\u00e4 t\u00e4n\u00e4\u00e4n iltap\u00e4iv\u00e4ll\u00e4, mutta pettymyksekseni neuvoston edustaja ei ole t\u00e4\u00e4ll\u00e4. Komissiollakin on selke\u00e4 teht\u00e4v\u00e4, ja minulla on kaksi erityiskysymyst\u00e4 sille: suostuuko se suorittamaan tutkimuksen LFAS:n vaikutuksista meriymp\u00e4rist\u00f6\u00f6n ja takaako se, ett\u00e4 aihe otetaan esille transatlanttisessa vuoropuhelussa? PARSEP Odotan mielenkiinnolla vastaustanne. PARSEP Keskustelu on p\u00e4\u00e4ttynyt. PARSEP Esityslistalla on seuraavana keskustelu ihmisoikeuksien sek\u00e4 demokratian ja oikeusvaltion periaatteiden loukkauksia koskevista tapauksista (50 artikla). PARSEP Esityslistalla on seuraavana yhteiskeskustelu seuraavista kuudesta Pohjois-Korean humanitaarista kriisi\u00e4 koskevasta p\u00e4\u00e4t\u00f6slauselmaesityksest\u00e4: PARSEP B5-0036/2003 Ford ja Van den Berg PSE-ryhm\u00e4n puolesta: Humanitaarinen tilanne Korean demokraattisessa kansantasavallassa, PARSEP B5-0039/2003 Tannock, Gawronski, Perry ja Maij-Weggen PPE-DE-ryhm\u00e4n puolesta: Humanitaarinen kriisi Pohjois-Koreassa, PARSEP B5-0043/2003 Watson, Malmstr\u00f6m ja Andreasen ELDR-ryhm\u00e4n puolesta: N\u00e4l\u00e4nh\u00e4t\u00e4 Pohjois-Koreassa, PARSEP B5-0045/2003 Lagendijk, Turmes, Ahern, Gahrton, Schroedter, Wuori ja Maes Verts/ALE-ryhm\u00e4n puolesta: Pohjois-Korea, PARSEP B5-0048/2003 Belder EDD-ryhm\u00e4n puolesta: Humanitaarinen kriisi Pohjois-Koreassa, PARSEP B5-0051/2003 Vinci GUE/NGL-ryhm\u00e4n puolesta: Humanitaarinen tilanne Korean demokraattisessa kansantasavallassa. PARSEP Arvoisa puhemies, hyv\u00e4t kollegat, hyv\u00e4t parlamentin j\u00e4senet, olemme eritt\u00e4in huolestuneita Pohjois-Korean tapahtumista - maa, jossa ei ole demokratiaa, jossa kansan annetaan n\u00e4hd\u00e4 n\u00e4lk\u00e4\u00e4 ja jossa t\u00e4st\u00e4 huolimatta katsotaan olevan v\u00e4ltt\u00e4m\u00e4t\u00f6nt\u00e4 puolustautua \"ydinaseilla\". Meid\u00e4n on eritt\u00e4in vaikea l\u00f6yt\u00e4\u00e4 toimintalinja, jossa osoitetaan syv\u00e4\u00e4 huolestuneisuutta Pohjois-Korean tilanteesta pahentamatta konfliktia. Emme halua luoda samanlaista tilannetta kuin Irakissa, mutta haluamme painostaa Pohjois-Korean hallitusta voimakkaasti, jotta se keskittyisi kansansa tarpeisiin. PARSEP Loppujen lopuksi t\u00e4m\u00e4 ei ole ensimm\u00e4inen kerta, kun olemme kiinnitt\u00e4neet huomiota t\u00e4h\u00e4n kysymykseen. On eritt\u00e4in valitettavaa, ett\u00e4 olemme monesti l\u00e4hestyneet Pohjois-Koreaa korostaaksemme ydinenergian yksinomaan rauhanomaista k\u00e4ytt\u00f6\u00e4, jotta maan v\u00e4h\u00e4isi\u00e4 resursseja voidaan aidosti kehitt\u00e4\u00e4. PARSEP T\u00e4m\u00e4n takia haluamme j\u00e4lleen kerran todeta selv\u00e4sti, ett\u00e4 haluamme saada aikaan vuoropuhelua, ja kehotamme Pohjois-Koreaa ryhtym\u00e4\u00e4n vuoropuheluun kanssamme. Olemme tyytyv\u00e4isi\u00e4, ett\u00e4 amerikkalaiset ovat my\u00f6s valmiita ainakin t\u00e4ss\u00e4 tapauksessa k\u00e4ym\u00e4\u00e4n vuoropuhelua rauhanomaisen ratkaisun saavuttamiseksi ja ett\u00e4 neuvotteluja k\u00e4yd\u00e4\u00e4n parhaillaan. Pyyd\u00e4mme komissiota jatkamaan humanitaarisen avun antamista huolimatta n\u00e4ist\u00e4 ep\u00e4miellytt\u00e4vist\u00e4 poliittisista olosuhteista, joita ei voida hyv\u00e4ksy\u00e4, ja auttamaan maan n\u00e4lk\u00e4\u00e4n\u00e4kev\u00e4\u00e4 v\u00e4est\u00f6\u00e4. Arvoisa puhemies, kun t\u00e4m\u00e4 pidet\u00e4\u00e4n mieless\u00e4 katson, ett\u00e4 olemme valmiita tukemaan asiaa koskevaa p\u00e4\u00e4t\u00f6slauselmaa ja ilmaisemaan toiveen, ett\u00e4 Yhdysvallat ja Eurooppa p\u00e4\u00e4tt\u00e4v\u00e4t t\u00e4ss\u00e4 tapauksessa yhteisest\u00e4 l\u00e4hestymistavasta, jotta Pohjois-Korea saadaan auttamaan omaa kansaansa sen sijaan, ett\u00e4 se uhkaa globaalia yhteis\u00f6\u00e4. PARSEP Arvoisa puhemies, samalla, kun Pohjois-Korean diktaattori, Kim Jong-Il siemaisee h\u00e4nt\u00e4 varten maahantuotua ranskalaista konjakkia, maistelee kahden italialaisen kokin \u00e4skett\u00e4in paistamia pizzoja ja harkitsee, mit\u00e4 sanoa YK:n l\u00e4hettil\u00e4\u00e4lle, monet tuhannet sorretut alamaiset, my\u00f6s vauvat, kuolevat kirjaimellisesti n\u00e4lk\u00e4\u00e4n, sill\u00e4 h\u00e4nen harjoittamansa hirvitt\u00e4v\u00e4 stalinistinen talouspolitiikka on saattanut maan polvilleen. PARSEP T\u00e4t\u00e4kin pahempaa on, ett\u00e4 lukuisia kansalaisia on l\u00e4hetetty poliittisina vankeina uudelleenkoulutusleireille, joissa heit\u00e4 kidutetaan ja pahoinpidell\u00e4\u00e4n. On tyrmistytt\u00e4v\u00e4\u00e4, ett\u00e4 t\u00e4m\u00e4 maa kutsuu itse\u00e4\u00e4n \"Korean demokraattiseksi kansantasavallaksi\", ja silti kansalaisia ei ole koskaan kuultu, vaan Kim peri valtansa edesmenneelt\u00e4 diktaattori-is\u00e4lt\u00e4\u00e4n. PARSEP Maa, jossa vallitsee julma ja salaileva kommunistinen diktatuuri, on toistuvasti rikkonut kansainv\u00e4lisi\u00e4 sopimuksia, kuten vuoden 1994 sopimusta, jonka tarkoituksena oli p\u00e4\u00e4tt\u00e4\u00e4 ydinpommin rakentamisessa tarvittavan uraanin rikastamista koskeva ohjelma. Se on my\u00f6s aiemmin rikkonut sivistyneen maailman jokaista s\u00e4\u00e4nt\u00f6\u00e4. Se on muun muassa kaapannut japanilaisia siviilej\u00e4 - mink\u00e4 hallitus perusteli vakoojien kieltenopettajien tarpeella, mik\u00e4 on aivan uskomatonta - ja osallistunut poliittisiin murhiin, kuten Etel\u00e4-Korean kabinetin r\u00e4j\u00e4ytt\u00e4miseen Rangoonissa muutama vuosi sitten. Se oli my\u00f6s vastuussa etel\u00e4korealaisen lentokoneen r\u00e4j\u00e4ytt\u00e4misest\u00e4. PARSEP Pohjois-Korea on \u00e4skett\u00e4in - ja t\u00e4m\u00e4 on p\u00e4\u00e4t\u00f6slauselmaesitykseni syy - antanut useiden tuhansien ihmisten kuolla n\u00e4lk\u00e4\u00e4n sen tuhoisan erist\u00e4ytyneen talouspolitiikan takia ja koska se on halunnut yll\u00e4pit\u00e4\u00e4 miljoonan hengen armeijaa, jolloin elintarvikkeita ja resursseja on ohjattu ep\u00e4moraalisesti armeijalle, vaikka siviiliv\u00e4est\u00f6 k\u00e4rsii n\u00e4l\u00e4nh\u00e4d\u00e4st\u00e4. Se on eritt\u00e4in ep\u00e4onnistunut h\u00e4irikk\u00f6valtio ja sellaisena eritt\u00e4in vaarallinen. Se voisi aiheuttaa tuhoa Etel\u00e4-Koreassa, jonka kanssa se on teknisesti sodassa. PARSEP N\u00e4in ollen EU:n on edett\u00e4v\u00e4 varovasti ja keskitytt\u00e4v\u00e4 humanitaarisen avun antamiseen Pohjois-Korean kauan k\u00e4rsineelle kansalle. T\u00e4st\u00e4 huolimatta avunantajien on yh\u00e4 vaadittava varmennusmenettelyj\u00e4 sen varmistamiseksi, ett\u00e4 elintarvikkeet luovutetaan niille, joille ne on tarkoitettu eik\u00e4 niit\u00e4 anneta armeijalle tai myyd\u00e4 edelleen, mik\u00e4 on viel\u00e4kin pahempaa, jotta saadaan rahaa aseiden tai poliittiselle ja sotilaalliselle eliitille tarkoitettujen ylellisyyksien ostoon. PARSEP Pohjois-Korea on aikamme suurin humanitaarinen skandaali. Sill\u00e4 aikaa kun \"suuri johtaja\" el\u00e4\u00e4 ylellisyydess\u00e4, h\u00e4n antaa kirjaimellisesti oman kansansa kuolla. Yli kahden miljoonan ihmisen v\u00e4itet\u00e4\u00e4n kuolleen n\u00e4lk\u00e4\u00e4n viimeisen viiden vuoden aikana, ja yli 13 miljoonaa ihmist\u00e4 k\u00e4rsii aliravitsemuksesta. Kommunistisella huonolla johdolla on varmistettu, ett\u00e4 aliravittu kansa j\u00e4tet\u00e4\u00e4n ulos kylm\u00e4\u00e4n ja ett\u00e4 heilt\u00e4 ev\u00e4t\u00e4\u00e4n asianmukainen terveydenhoito. T\u00e4ss\u00e4 ty\u00f6ntekij\u00f6iden paratiisissa noin 200 000:ta ihmist\u00e4 pidet\u00e4\u00e4n poliittisista syist\u00e4 vangittuna uudelleenkoulutusleireill\u00e4. Lis\u00e4ksi Kiina l\u00e4hett\u00e4\u00e4 Pohjois-Koreasta saapuvat poliittiset turvapaikanhakijat j\u00e4rjestelm\u00e4llisesti takaisin, vaikka se tiet\u00e4\u00e4 eritt\u00e4in hyvin, ett\u00e4 palautettuja rangaistaan ep\u00e4inhimillisesti. PARSEP Ydinasepelill\u00e4 Kim Jong-Il haastaa maailman ja yritt\u00e4\u00e4 est\u00e4\u00e4 kansainv\u00e4list\u00e4 yhteis\u00f6\u00e4 vastaamasta asianmukaisesti. Ei ole selv\u00e4\u00e4, onko Kimille mahdollista puhua j\u00e4rkev\u00e4sti. Yhdysvaltain asenne Pohjois-Koreaa kohtaan on esimerkillisen varovainen verrattuna sen asenteeseen Irakia kohtaan. Euroopan unionin ydinuhkaa koskevan linjan on oltava yhdenmukainen Yhdysvaltain linjan kanssa - mik\u00e4li sill\u00e4 on jokin linja. Arvaamattoman diktaattorin k\u00e4siss\u00e4 massatuhoaseet ovat pysyv\u00e4 uhka maailmanrauhalle, eik\u00e4 t\u00e4t\u00e4 voida sallia. PARSEP Meid\u00e4n on keskitytt\u00e4v\u00e4 my\u00f6s humanitaariseen apuun, jolla est\u00e4mme inhimillist\u00e4 k\u00e4rsimyst\u00e4 saamasta yh\u00e4 suuremmat mittasuhteet. On hyv\u00e4 asia, ett\u00e4 komissio aikoo varata 9,5 miljoonaa euroa naisille ja lapsille jo k\u00e4ytetyn 50 miljoonan euron lis\u00e4ksi. Kysymys kuuluu, onko t\u00e4m\u00e4 tarpeeksi. Oletan, ett\u00e4 komissio on tarvittaessa valmis tekem\u00e4\u00e4n enemm\u00e4n. Olisin kiitollinen, jos komissio voisi vahvistaa t\u00e4m\u00e4n. Pohjois-Korean tilanne on ajautunut ojasta allikkoon sen itse aiheutetun eristyneisyyden takia. Meid\u00e4n on l\u00f6ydett\u00e4v\u00e4 keinot t\u00e4m\u00e4n est\u00e4miseksi ja aikamme suurimman humanitaarisen skandaalin lopettamiseksi. PARSEP Arvoisa puhemies, Pohjois-Koreassa on parhaillaan meneill\u00e4\u00e4n vakava humanitaarinen kriisi. V\u00e4h\u00e4isten tai puuttuvien energial\u00e4hteiden ja elintarvikepulan yhdistelm\u00e4 on tuhoisa ankarissa talviolosuhteissa. T\u00e4st\u00e4 syyst\u00e4 Verts/ALE-ryhm\u00e4 on tyytyv\u00e4inen komission p\u00e4\u00e4t\u00f6kseen antaa Pohjois-Korealle elintarvikeapua 9,5 miljoonan euron arvosta. PARSEP Meid\u00e4n on kuitenkin harkittava perusteellisesti uudelleen energial\u00e4hteiden turvaamista koskevaa l\u00e4hestymistapaamme. Korean energia-alan kehitysj\u00e4rjest\u00f6 KEDO ei ollut sopiva t\u00e4ytt\u00e4m\u00e4\u00e4n maan tarpeita ja toteuttamaan infrastruktuuria, eik\u00e4 tilanteen voida antaa jatkua. Tuomitsimme sen, ja energiakriisi\u00e4 koskeva Euroopan unionin l\u00e4hestymistapa on uusittava kokonaan. Olin tyytyv\u00e4inen kuullessani komission j\u00e4senen Pattenin ilmoittavan, ett\u00e4 Euroopan unionin on kehitett\u00e4v\u00e4 vaihtoehtoinen energiaohjelma Pohjois-Korealle. Suhtaudumme my\u00f6nteisesti komission kantaan ja odotamme mielenkiinnolla sopivan ohjelman kehitt\u00e4mist\u00e4. PARSEP Meid\u00e4n on tarkasteltava uudelleen yhteisty\u00f6politiikkaamme Pohjois-Korean kanssa k\u00f6yhyyden v\u00e4hent\u00e4misohjelman puitteissa markkinatalouden, demokraattisen uudistusten ja ihmisoikeuksien kunnioittamisen vaikutusten tasapainottamiseksi, ja Euroopan unionin ja muiden valtioiden on harkittava uudelleen Pohjois-Korean virallista diplomaattista tunnustamista. T\u00e4m\u00e4n seurauksena Pohjois-Korealla olisi mahdollista saada varoja Kansainv\u00e4lisest\u00e4 valuuttarahastosta (IMF), Aasian kehityspankista ja muista rahoituslaitoksista. PARSEP Kuten Swoboda totesi aiemmin t\u00e4m\u00e4n keskustelun aikana, t\u00e4rkeint\u00e4 on jatkaa neuvotteluja ja ennen kaikkea pyrki\u00e4 kriisin rahanomaiseen ratkaisuun. PARSEP Arvoisa puhemies, t\u00e4n\u00e4 iltap\u00e4iv\u00e4n\u00e4 laskeudumme yhteen kylm\u00e4n sodan viimeisimmist\u00e4 synkist\u00e4 vankityrmist\u00e4. Maapallon 38. leveysasteella sijaitsee \"rautaverho\", joka on tiiviimpi kuin Berliinin muuri koskaan. Sen takana \"suuren johtajan\" Kim Il-Sungin j\u00e4lkeen \"arvoisa johtaja\" Kim Jong-Il pelaa nyt masentavaa peli\u00e4, jota kutsutaan nimelt\u00e4 \"suuri valhe\". K\u00e4yt\u00e4nn\u00f6llisesti katsoen dynastisessa vallanperimyksess\u00e4 is\u00e4 ja poika ovat uhranneet Pohjois-Korean kansan suuruudenhullulle itsens\u00e4 ylent\u00e4miselle ja henkil\u00f6kohtaiselle diktatuurille yli puolen vuosisadan ajan. Pjongjang on pitk\u00e4\u00e4n yritt\u00e4nyt kielt\u00e4\u00e4 t\u00e4m\u00e4n kauhean todellisuuden. Pakenemaan onnistuneiden pohjoiskorealaisten todisteet on hyl\u00e4tty valheina. Satelliittikuvat antavat kuitenkin kiistattomat todisteet: poliittisia vankeja pidet\u00e4\u00e4n todellakin vangittuina orjaleireill\u00e4. PARSEP Silm\u00e4mme ovat avautuneet eritt\u00e4in my\u00f6h\u00e4\u00e4n n\u00e4kem\u00e4\u00e4n Pohjois-Korean hallituksen aiheuttaman vakavan vaaran my\u00f6s kansainv\u00e4liselle yhteis\u00f6lle. V\u00e4lit\u00f6n syy t\u00e4lle oli Pjongjangin \u00e4skett\u00e4inen yksipuolinen vet\u00e4ytyminen ydinsulkusopimuksesta. On mahdollista, ett\u00e4 Pohjois-Korea kehitt\u00e4\u00e4 ydinaseita ja myy n\u00e4it\u00e4 aseita sek\u00e4 ohjusj\u00e4rjestelmi\u00e4 laittomasti, mik\u00e4 edellytt\u00e4\u00e4 nyt pikaista vastausta kansainv\u00e4liselt\u00e4 yhteis\u00f6lt\u00e4. PARSEP Pyyd\u00e4n siksi neuvostoa ja komissiota korostamaan alueellista diplomaattista l\u00e4hestymistapaa Pohjois-Korean kriisiin. Mainittakoon, ett\u00e4 kansainv\u00e4lisen yhteis\u00f6n olisi p\u00e4\u00e4tt\u00e4v\u00e4isesti hyl\u00e4tt\u00e4v\u00e4 Pjongjangin m\u00e4\u00e4r\u00e4ys - hy\u00f6kk\u00e4\u00e4m\u00e4tt\u00f6myyssopimus Yhdysvaltojen kanssa, joka otetaan k\u00e4ytt\u00f6\u00f6n ydinaseuhan takia - my\u00f6t\u00e4tunnosta Kimin \"rautaverhon\" takana olevia Pohjois-Korean kansalaisia kohtaan, joiden on elett\u00e4v\u00e4 olosuhteissa, joita on melkein mahdoton kuvitella, ja my\u00f6s heid\u00e4n turvallisuutensa takia. PARSEP Arvoisa puhemies, t\u00e4\u00e4ll\u00e4 on mainittu Pohjois-Korean lukuisat ihmisoikeusloukkaukset, jotka loukkaavat meid\u00e4n moraalik\u00e4sityst\u00e4mme ja sivilisaation k\u00e4sitett\u00e4mme. Vierailtuani maassa kolmesti katson, ett\u00e4 kaikkein poikkeuksellisin ja hirvitt\u00e4vin ihmisoikeusloukkaus on se, ett\u00e4 jos Pohjois-Koreassa pidett\u00e4isiin vapaat demokraattiset vaalit, kuten meill\u00e4 on tapana, nykyinen johtaja Kim Jong-Il voittaisi ne. T\u00e4m\u00e4 osoittaa ensinn\u00e4kin Pohjois-Korean johtajien hirvitt\u00e4v\u00e4\u00e4 kyky\u00e4 pit\u00e4\u00e4 maa eristettyn\u00e4 ulkoisilta vaikutuksilta - ja nykyisess\u00e4 globaalistuneessa maailmassa t\u00e4m\u00e4 ei ole helppoa - ja toiseksi antautumisen ja n\u00f6yryytyksen laajuutta, jolle v\u00e4est\u00f6 on alistettu. Ihmisoikeusrikkomukset ja v\u00e4kivalta eiv\u00e4t kohdistu ainoastaan heid\u00e4n ruumiinsa n\u00e4l\u00e4n ja keskitysleirien muodossa, vaan my\u00f6s heid\u00e4n mieleens\u00e4, sill\u00e4 heid\u00e4t on aivopesty, eiv\u00e4tk\u00e4 he pysty ajattelemaan omilla aivoillaan, kuten Ven\u00e4j\u00e4n kulakit. Pohjoiskorealaisista on tehty lampaita, jotka seuraavat johtajaansa sokeasti. PARSEP T\u00e4t\u00e4 sivistym\u00e4t\u00f6nt\u00e4 ja samalla groteskia hallintoa on hillitt\u00e4v\u00e4. Olen yksi niist\u00e4, jotka katsovat, ett\u00e4 Yhdysvaltain l\u00e4hestymistapa on erityisesti viime p\u00e4ivin\u00e4 ollut oikea: sen vaatimukset ovat maltillisempia, se on v\u00e4ltt\u00e4nyt kriisin Etel\u00e4-Korean kanssa ja l\u00e4hestynyt Kiinaa, jolla on keskeinen asema alueen kriisin ratkaisemisessa. Euroopan panos, meid\u00e4n panoksemme on valitettavasi vain v\u00e4h\u00e4inen poliittisella rintamalla. Voimme ja meid\u00e4n t\u00e4ytyy tehd\u00e4 enemm\u00e4n humanitaarisella tasolla: meid\u00e4n on tietenkin l\u00e4hetett\u00e4v\u00e4 apua, mutta meid\u00e4n on my\u00f6s varmistettava (kuten on todettu monta kertaa), ett\u00e4 apu menee oikeisiin kohteisiin eli apua tarvitseville. Asiaa koskevista julkilausumista ja henkil\u00f6kohtaisista lausumista huolimatta en ole lainkaan varma, ett\u00e4 t\u00e4m\u00e4 voidaan taata t\u00e4n\u00e4\u00e4n. Haluan todella t\u00e4t\u00e4 koskevan vakuutuksen ja haluaisin, ett\u00e4 komissio voisi antaa meille sen. PARSEP Arvoisa puhemies, kollegani Chris Patten on useammin kuin kerran puhunut parlamentissa Koreasta, ja h\u00e4nell\u00e4 on tapana viitata Pohjois-Koreaan outona maana. PARSEP Jotkut teist\u00e4 ovat tuoneet esiin Pohjois-Korean ristiriitaisuudet. Toisaalta maalla on valtava armeija, sill\u00e4 on salainen uraanin rikastamisohjelma, se on ensimm\u00e4inen ydinsulkusopimuksesta vet\u00e4ytynyt maa ja se uhkaa naapureitaan ja k\u00e4rsii toisaalta vakavista sis\u00e4isist\u00e4 ongelmista. PARSEP Kylm\u00e4n sodan p\u00e4\u00e4ttyminen lakkautti maan halvat \u00f6ljy- ja raaka-ainel\u00e4hteet, eik\u00e4 se voi tuottaa tarpeeksi maan tarpeiden tyydytt\u00e4miseksi. L\u00e4\u00e4kkeist\u00e4 on pula, energiaa ei ole tarpeeksi jne. PARSEP Miten Euroopan unioni voi vastata n\u00e4in ristiriitaiseen tilanteeseen? On selv\u00e4\u00e4, ettemme voi jatkossa tehd\u00e4 asioita, kuten ne on tehty t\u00e4h\u00e4n asti. T\u00e4m\u00e4n takia Korean niemimaan energia-alan kehitysj\u00e4rjest\u00f6 p\u00e4\u00e4tti viime marraskuussa keskeytt\u00e4\u00e4 raskaan polttoaineen toimituksen Pohjois-Koreaan. Kuten jotkut teist\u00e4 ovat ehdottaneet, mielest\u00e4ni on t\u00e4rke\u00e4\u00e4, ett\u00e4 samanaikaisesti jatkamme toimiamme ja vuoropuhelua, jotta Pohjois-Korea tulee j\u00e4rkiins\u00e4. Keskustelemme parhaillaan unionin j\u00e4senvaltioiden kanssa parhaasta tavasta t\u00e4m\u00e4n tavoitteen saavuttamiseksi ja olemme my\u00f6s tyytyv\u00e4isi\u00e4 siihen, ett\u00e4 Yhdysvallat pohtii joitakin n\u00e4ist\u00e4 ongelmista vuoropuhelun k\u00e4ynnist\u00e4miseksi Pohjois-Korean kanssa. PARSEP On totta, ett\u00e4 vaikka olemme huolissamme Korean demokraattisen kansantasavallan tapahtumista, emme voi emmek\u00e4 halua rangaista Korean heikointa ja v\u00e4h\u00e4osaisinta v\u00e4est\u00f6nosaa varsinkaan sen hallituksen tekemist\u00e4 virheist\u00e4. Jotkut teist\u00e4 ovat puhuneet groteskeista hallituksista, jotka eiv\u00e4t reagoi maassa vallitsevaan todellisuuteen. PARSEP Siksi neuvostossa marraskuussa pidettyjen keskustelujen j\u00e4lkeen ministerit olivat yht\u00e4 mielt\u00e4 siit\u00e4, ett\u00e4 meid\u00e4n on jatkettava humanitaarisen avun ja elintarvikeavun toimittamista. Olette ehdottaneet, ett\u00e4 meid\u00e4n on jatkettava avun toimittamista, mutta varoitatte my\u00f6s, ett\u00e4 meid\u00e4n on valvottava huolellisemmin avun toimittamista. Neuvosto on t\u00e4ysin samaa mielt\u00e4. Humanitaarisen avun riitt\u00e4v\u00e4 valvonta on n\u00e4in ollen ensisijainen tavoite, ja harkitsemme t\u00e4t\u00e4 kysymyst\u00e4 erityisesti sek\u00e4 kahdenv\u00e4lisiss\u00e4 yhteyksiss\u00e4 Korean demokraattisen kansantasavallan kanssa ett\u00e4 maailman elintarvikeohjelman yhteydess\u00e4, jonka kautta apu jaetaan kyseiseen maahan. PARSEP Joistakin k\u00e4ytt\u00e4mist\u00e4mme tiedoista voi olla hy\u00f6ty\u00e4 parlamentille. Kuten tied\u00e4tte, ECHO on t\u00e4rkein humanitaarisen avun ulkomainen antaja Korean demokraattiselle tasavallalle, jos elintarvikeapua ei oteta huomioon, ja vuodesta 1995 se on antanut humanitaarista apua yli 62 miljoonan euron arvosta. Apua annetaan tasapuolisesti. Sit\u00e4 annetaan eniten tarvitseville Euroopan unionin kansalaisj\u00e4rjest\u00f6jen, Punaisen Ristin ja Yhdistyneiden Kansakuntien j\u00e4rjest\u00f6jen v\u00e4lityksell\u00e4. PARSEP Vuonna 2002 my\u00f6nnettiin 11,5 miljoonaa euroa vesivarastojen, kaivojen ja puhtaanapitorakenteiden kunnostamiseen ja suojeluun sek\u00e4 muun muassa vesiastioiden, klooritablettien, henkil\u00f6kohtaiseen hygieniaan liittyvien tavaroiden ja l\u00e4\u00e4kkeiden jakeluun. PARSEP Yhteis\u00f6 ei ole t\u00e4ll\u00e4 hetkell\u00e4 kovinkaan merkitt\u00e4v\u00e4 elintarvikeavun toimittaja. Kuten jotkut teist\u00e4 ovat huomauttaneet, vuonna 2002 toimitimme 23 000 tonnia vehn\u00e4\u00e4 5 miljoonan euron arvosta maailman elintarvikeohjelmalle Pohjois-Koreaan. PARSEP Viime viikolla hyv\u00e4ksyimme elintarvikeh\u00e4t\u00e4apupaketin, jonka arvo on 9,5 miljoonaa euroa. Se sis\u00e4lt\u00e4\u00e4 39 000 tonnia viljaa, joka oli tarkoitettu erityisesti lapsille vastauksena Pohjois-Korean maailman elintarvikeohjelmassa marraskuussa tehtyyn vetoomukseen. ECHO my\u00f6nsi avustuksen sill\u00e4 edellytyksell\u00e4, ett\u00e4 - ja olemme j\u00e4lleen huolissamme valvontaa koskevasta ongelmasta - Korean demokraattinen tasavalta hyv\u00e4ksyy maailman elintarvikeohjelmassa esitetyn pyynn\u00f6n; t\u00e4m\u00e4 merkitsisi parempaa valvontaa ja paikalla suoritettavia tarkastuksia. Viimeisimm\u00e4n pyynn\u00f6n osalta Italia, Norja ja Sveitsi ovat ainoat maat, jotka ovat sitoutuneet etenem\u00e4\u00e4n t\u00e4t\u00e4 tiet\u00e4 komission kanssa. PARSEP Keskustelu on p\u00e4\u00e4ttynyt. PARSEP \u00c4\u00e4nestys toimitetaan, kun iltap\u00e4iv\u00e4n keskustelut ovat p\u00e4\u00e4ttyneet. PARSEP Esityslistalla on seuraavana yhteiskeskustelu viidest\u00e4 P\u00e4iv\u00e4ntasaajan Guineaa koskevasta p\u00e4\u00e4t\u00f6slauselmaesityksest\u00e4: PARSEP B5-0038/2003 Sauquillo P\u00e9rez del Arco, Carlotti, Mart\u00ednez Mart\u00ednez, Obiols i Germ\u00e0 ja Carnero Gonz\u00e1lez PSE-ryhm\u00e4n puolesta: presidentinvaalit ja ihmisoikeusrikkomukset P\u00e4iv\u00e4ntasaajan Guineassa PARSEP B5-0040/2003 Ferrer ja Posselt PPE-DE-ryhm\u00e4n puolesta: P\u00e4iv\u00e4ntasaajan Guinea PARSEP B5-0044/2003 Sanders-ten Holte, S\u00e1nchez Garc\u00eda ja Gas\u00f2liba i B\u00f6hm ELDR-ryhm\u00e4n puolesta: ihmisoikeustilanne P\u00e4iv\u00e4ntasaajan Guineassa PARSEP B5-0046/2003 Rod, Isler B\u00e9guin ja Maes Verts/ALE-ryhm\u00e4n puolesta: presidentinvaalit P\u00e4iv\u00e4ntasaajan Guineassa PARSEP B5-0050/2003 Marset Campos ja Gonz\u00e1lez \u00c1lvarez GUE/NGL-ryhm\u00e4n puolesta: presidentinvaalit ja ihmisoikeusrikkomukset P\u00e4iv\u00e4ntasaajan Guineassa. PARSEP Arvoisa puhemies, P\u00e4iv\u00e4ntasaajan Guinean viranomaisten jo vuosia osoittama piittaamattomuus demokratiaa kohtaan yltyi huippuunsa \u00e4skett\u00e4isiss\u00e4 joulukuussa pidetyiss\u00e4 vaaleissa. PARSEP Vaalitulos, jonka mukaan hallitus sai taakseen 97 prosenttia \u00e4\u00e4nist\u00e4, ei perustu \u00e4\u00e4ntenlaskentaan, ja vaalien yhteydess\u00e4 tehty v\u00e4est\u00f6nlaskenta her\u00e4tti ep\u00e4ilyksi\u00e4. Lis\u00e4ksi oppositiojohtajat olivat vangittuina eik\u00e4 heill\u00e4 siten ollut sananvapautta koko vaalikampanjan aikana tai mahdollisuutta suljettuun \u00e4\u00e4nestykseen vaalip\u00e4iv\u00e4n\u00e4, sill\u00e4 \u00e4\u00e4nestyspaikat olivat armeijan hallinnassa. PARSEP Ik\u00e4vien tapahtumien ja v\u00e4kivaltaisuuksien vuoksi oppositiopuolueet lopulta vet\u00e4ytyiv\u00e4t vaaleista. Voidaan siis yksinkertaisesti todeta, ett\u00e4 vaalit olivat vilpilliset ja p\u00e4tem\u00e4tt\u00f6m\u00e4t. Euroopan unionin olisikin n\u00e4ytett\u00e4v\u00e4 esimerkki\u00e4, ja osoitettava, ettei kansainv\u00e4linen yhteis\u00f6 hyv\u00e4ksy vaalitulosta. PARSEP Arvoisa puhemies, PSE-ryhm\u00e4 huomauttaa, ett\u00e4 kyseess\u00e4 on Euroopan parlamentin viides p\u00e4\u00e4t\u00f6slauselma ihmisoikeustilanteesta P\u00e4iv\u00e4ntasaajan Guineassa. Ryhm\u00e4 ottaa huomioon my\u00f6s unionin puheenjohtajavaltion julkilausuman P\u00e4iv\u00e4ntasaajan Guinean vaaleista ja pyyt\u00e4\u00e4 t\u00e4ten, ett\u00e4 Euroopan unionin toimielimet tekev\u00e4t kaikkensa varmistaakseen, ett\u00e4 P\u00e4iv\u00e4ntasaajan Guineassa aloitetaan rahanomainen siirtyminen demokratiaan, ett\u00e4 poliittiset vangit vapautetaan v\u00e4litt\u00f6m\u00e4sti ja ett\u00e4 presidentti Obiangin tekemi\u00e4 kansainv\u00e4lisi\u00e4 sitoumuksia noudatetaan. PARSEP Arvoisa puhemies, PSE-ryhm\u00e4 pit\u00e4\u00e4 erityisen t\u00e4rke\u00e4n\u00e4 yhteisen p\u00e4\u00e4t\u00f6slauselman 7 kohtaa, jossa komissiota pyydet\u00e4\u00e4n laatimaan kertomus P\u00e4iv\u00e4ntasaajan Guinean alueella toimivien \u00f6ljy-yhti\u00f6iden toiminnan vaikutuksista demokratiaan ja ihmisoikeuksiin. Kyseess\u00e4 on maa, jolla on \u00f6ljy\u00e4. \u00d6ljyst\u00e4 saatavat tulot menev\u00e4t presidentti Obiangille, h\u00e4nen perheelle ja yst\u00e4ville, mutta eiv\u00e4t Guinean kansalaisille. Guinean kansalaisista kolmanneksen odotettu elinik\u00e4 on 40 vuotta ja 57 prosentilla ei ole juomavett\u00e4. PARSEP Arvoisa puhemies, ehk\u00e4p\u00e4 YK:n turvallisuusneuvoston pit\u00e4isikin t\u00e4m\u00e4n vuoksi julistaa P\u00e4iv\u00e4ntasaajan Guinea \u00f6ljysaartoon ja ehk\u00e4 nyt, kun turvallisuusneuvostossa on nelj\u00e4 Euroopan unionin j\u00e4senvaltioiden edustajaa, turvallisuusneuvosto voisi sopia demokratisointiprosessien valvonnasta tietyiss\u00e4 maissa. Lis\u00e4ksi turvallisuusneuvosto voisi antaa Euroopan unionin ensisijaisten tavoitteiden mukaisesti tukea menetelmiin, joita v\u00e4ltt\u00e4m\u00e4tt\u00f6m\u00e4t muutokset edellytt\u00e4v\u00e4t. PARSEP Meid\u00e4n on sanottava \"jo riitt\u00e4\u00e4\" sellaisille maille, joissa ainoastaan v\u00e4hemmist\u00f6 kansasta hy\u00f6tyy voimavaroista ja enemmist\u00f6 el\u00e4\u00e4 edelleen k\u00f6yhyydess\u00e4. PARSEP Arvoisa puhemies, t\u00e4m\u00e4n vuoksi p\u00e4\u00e4t\u00f6slauselma on mielest\u00e4mme eritt\u00e4in hyv\u00e4 ja my\u00f6nteinen silt\u00e4 osin, ett\u00e4 sill\u00e4 edistet\u00e4\u00e4n parlamentin ja Euroopan unionin ensisijaisen tavoitteen saavuttamista. PARSEP Arvoisa puhemies, arvoisa komission j\u00e4sen, olemme eritt\u00e4in huolestuneita ihmisoikeustilanteesta ja demokratisoitumisen hitaudesta P\u00e4iv\u00e4ntasaajan Guineassa. PARSEP Viime kev\u00e4\u00e4n\u00e4 j\u00e4rjestetty oikeudenk\u00e4ynti, jossa tuomittiin lopulta 68 ihmist\u00e4, oli osoitus siit\u00e4, ettei maassa ole mink\u00e4\u00e4nlaisia kansainv\u00e4lisesti tuettavia oikeudellisia takeita. T\u00e4m\u00e4 k\u00e4vi selv\u00e4sti ilmi my\u00f6s julkilausumasta, jonka parlamentti antoi kes\u00e4kuussa. Oikeudenk\u00e4ynti keskeytti hitaan mutta my\u00f6nteisen kehityksen, ja siit\u00e4 on syyt\u00e4 antaa moitteita. Olisi kuitenkin muistettava, ettei yhteydenpidon katkaiseminen sellaisten maiden kanssa, joissa ihmisoikeuksien kunnioittaminen on suorastaan alij\u00e4\u00e4m\u00e4ist\u00e4, ole aina paras ratkaisu. PARSEP Kriittist\u00e4 vuoropuhelua on jatkettava, ja pyrkimyksi\u00e4 on edistett\u00e4v\u00e4 vaiheittaisesti. Ihmisoikeusrikkomukset on tuomittava ankarasti, mutta tehdyist\u00e4 edistysaskeleista on my\u00f6s annettava tunnustusta, kuten siit\u00e4, ett\u00e4 12. lokakuuta 34-vuotiaan itsen\u00e4isen Guinean juhlap\u00e4iv\u00e4n kunniaksi vapautettiin 120 pitk\u00e4\u00e4n vangittuna ollutta ihmist\u00e4. Tied\u00e4n, ettei t\u00e4m\u00e4 tietenk\u00e4\u00e4n riit\u00e4, mutta meid\u00e4n on annettava siit\u00e4 tunnustus Guinealle. PARSEP Demokratisoitumisen hitautta koskevaa kritiikki\u00e4 her\u00e4tt\u00e4\u00e4 se, ett\u00e4 nelj\u00e4 oppositioehdokasta joutui vet\u00e4ytym\u00e4\u00e4n vaaleista vaalip\u00e4iv\u00e4n\u00e4 \u00e4\u00e4nestysten ep\u00e4selvyyksien vuoksi, vaikka vaalikampanjan aikana ei ollut tapahtunut mit\u00e4\u00e4n vakavaa. Emme my\u00f6sk\u00e4\u00e4n saa unohtaa, ett\u00e4 vangittuna tai maanpaossa on edelleen joitakin suurten puolueiden j\u00e4seni\u00e4, kuten kielletyn edistyspuolueen puheenjohtaja Severo Moto, joka on maanpaossa Madridissa. PARSEP Uskomme, ett\u00e4 seuraavat vaalit, eli lains\u00e4\u00e4d\u00e4nt\u00f6elimen vaalit, jotka pidet\u00e4\u00e4n suunnitelmien mukaan muutaman kuukauden kuluttua, j\u00e4rjestet\u00e4\u00e4n varmemmissa ja vapaammissa olosuhteissa. Niinp\u00e4 t\u00e4n\u00e4\u00e4n k\u00e4sitelt\u00e4v\u00e4ss\u00e4 p\u00e4\u00e4t\u00f6slauselmassa olisikin kehotettava Guinean hallitusta toteuttamaan toimenpiteit\u00e4, joilla se voisi lis\u00e4t\u00e4 uskottavuuttaan j\u00e4rjestelm\u00e4n demokratisoinnissa. Lis\u00e4ksi Guineaa pit\u00e4isi kehottaa aloittamaan todellinen vuoropuhelu kaikkien poliittisten liikkeiden kanssa. Samanaikaisesti poliittisten liikkeiden olisi v\u00e4ltett\u00e4v\u00e4 puheita, jotka voivat aiheuttaa v\u00e4kivaltaisuuksia, jollaisia tapahtui nelj\u00e4n vaaleista vet\u00e4ytyneen ehdokkaan puheiden seurauksena. PARSEP Kritiikki\u00e4 on esitett\u00e4v\u00e4, jos sille on aihetta, mutta prosessia on my\u00f6s tuettava. Emme t\u00e4ten voi tukea p\u00e4\u00e4t\u00f6slauselmaa, sill\u00e4 se on suora hy\u00f6kk\u00e4ys Guinean hallitusta vastaan. Seuraukset voivat olla kielteisi\u00e4, eik\u00e4 se hy\u00f6dyt\u00e4 Guinean kansalaisia tai vie demokratisointiprosessia eteenp\u00e4in. PARSEP Vallan v\u00e4\u00e4rink\u00e4ytt\u00f6 kansalaisten kustannuksella on yleist\u00e4 Afrikassa. Jos presidentti valitaan 97 prosentin \u00e4\u00e4ntenenemmist\u00f6ll\u00e4, vaaleja voidaan pit\u00e4\u00e4 ep\u00e4onnistuneina. Niin kutsuttu taistelu vaaliuurnilla, jonka tuloksena presidentti Obiang sai pit\u00e4\u00e4 virkansa, oli farssi. Juuri sit\u00e4 saatoimme odottaakin hallitukselta, joka on vuosikausia j\u00e4tt\u00e4nyt huomiotta kansainv\u00e4lisen yhteis\u00f6n suositukset demokraattisen j\u00e4rjestelm\u00e4n luomisesta. Vaaleja tarkkailemaan l\u00e4hetetyt YK:n ja Euroopan unionin valtuuskunnat eiv\u00e4t olleet tervetulleita. Suurimman oppositiopuolueen p\u00e4\u00e4sihteeri pid\u00e4tettiin, ja oppositiojohtajille j\u00e4rjestettiin kev\u00e4\u00e4ll\u00e4 vilpilliset oikeudenk\u00e4ynnit. Ei siis ollut mitenk\u00e4\u00e4n yll\u00e4tt\u00e4v\u00e4\u00e4, ett\u00e4 istuvaa presidentti\u00e4 lukuun ottamatta kaikki ehdokkaat vet\u00e4ytyiv\u00e4t vaaleista. PARSEP P\u00e4iv\u00e4ntasaajan Guinea on yksi niist\u00e4 lukuisista k\u00f6yhyyden kurjistamista Afrikan maista, joissa pysyttelee vallassa yksi henkil\u00f6, joka rikastuttaa itse\u00e4\u00e4n, rikkoo ihmisoikeuksia ja j\u00e4tt\u00e4\u00e4 kansainv\u00e4lisen yhteis\u00f6n t\u00e4ysin huomiotta. Euroopan unionin on otettava tarmokkaasti osaa poliittiseen vuoropuheluun. Yhdeks\u00e4nnest\u00e4 Euroopan kehitysrahastosta ei voida osoittaa maalle lainkaan varoja, ellei tilanne parane merkitt\u00e4v\u00e4sti. Vaikka tilanne paranisikin demokratian ja hallintovalmiuksien kehitt\u00e4misen kannalta, nykyinen hallitus pit\u00e4isi tilannetta oman asemansa oikeuttajana. T\u00e4llaista poliittista hyv\u00e4ksik\u00e4ytt\u00f6\u00e4 ei voida hyv\u00e4ksy\u00e4. Oppositiojohtajat ja kaikki muutkin poliittiset vangit on vapautettava v\u00e4litt\u00f6m\u00e4sti. Vaalit on mit\u00e4t\u00f6it\u00e4v\u00e4, ja sen j\u00e4lkeen on j\u00e4rjestett\u00e4v\u00e4 uudet vaalit. \u00d6ljy-yhti\u00f6iden toiminnasta ja presidentin asemasta niiss\u00e4 on teht\u00e4v\u00e4 kansainv\u00e4linen tutkimus. PARSEP Demokratisointi on v\u00e4ltt\u00e4m\u00e4t\u00f6n edellytys, jotta k\u00f6yhyytt\u00e4 voidaan todella torjua. Maailmanlaajuistuminen, rajoitetut markkinat tai riitt\u00e4m\u00e4tt\u00f6m\u00e4t tuet eiv\u00e4t ole ainoita syit\u00e4 Afrikan maiden k\u00f6yhyyteen. Useat n\u00e4ist\u00e4 maista saavat asiansa sekaisin aivan itse. P\u00e4iv\u00e4ntasaajan Guinea on yksi niist\u00e4 maista, joissa vallan v\u00e4\u00e4rink\u00e4ytt\u00f6 jarruttaa kehityst\u00e4. Voi noita ihmispoloisia. PARSEP Arvoisa puhemies, P\u00e4iv\u00e4ntasaajan Guinea on valitettavasti jo vuosikymmeni\u00e4 ollut kielteinen esimerkki Afrikan maista, ja surullista kyll\u00e4 se ei ole ainoa t\u00e4llainen maa. Meid\u00e4n on my\u00f6nnett\u00e4v\u00e4 avoimesti, ett\u00e4 meill\u00e4 on merkitt\u00e4v\u00e4 vastuu t\u00e4st\u00e4 tilanteesta. Maassa on todellakin havaittu vallan v\u00e4\u00e4rink\u00e4ytt\u00f6\u00e4, korruptiota, ja jokaista oikeusvaltion luomisyrityst\u00e4 on seurannut diktatuurin v\u00e4\u00e4rink\u00e4ytt\u00f6\u00e4. T\u00e4m\u00e4 on t\u00e4ysin totta, ja t\u00e4llaiset toimet on tuomittava ankarasti. PARSEP Kuitenkin toisaalta my\u00f6s me olemme vastuussa tilanteesta, sill\u00e4 P\u00e4iv\u00e4ntasaaja Guinea on - kuten useat muutkin Afrikan maat - muuttunut voimavaroista k\u00e4yt\u00e4v\u00e4n kansainv\u00e4lisen kilpailun pelinappulaksi, ja valitettavasti kyseess\u00e4 ovat my\u00f6s Euroopan edut. Lis\u00e4ksi P\u00e4iv\u00e4ntasaajan Guinean tilannetta horjuttavat valitettavasti \u00f6ljy-yhti\u00f6iden suuri merkitys sek\u00e4 timanttikauppa ja laiton kauppa, kuten ihmiskauppa. Voimavaraoja suojellaan kyynisell\u00e4 politiikalla, jossa k\u00e4ytet\u00e4\u00e4n hyv\u00e4ksi etnisi\u00e4 ja poliittisia konflikteja. Yleens\u00e4 poliittisetkin konfliktit ovat kuitenkin saaneet alkunsa etnisist\u00e4 konflikteista. T\u00e4m\u00e4n vuoksi p\u00e4\u00e4t\u00f6slauselman 7 kohta on mielest\u00e4ni eritt\u00e4in t\u00e4rke\u00e4, sill\u00e4 meill\u00e4kin on vastuumme asiassa ja meill\u00e4 olisi my\u00f6s mahdollisuuksia k\u00e4ytt\u00e4\u00e4 vaikutusvaltaamme. Ehdotettu kertomus \u00f6ljy-yhti\u00f6iden asemasta on todellakin tarpeellinen. PARSEP En kannata ehdotusta, jonka mukaan maa olisi asetettava suoranaiseen boikottiin ja kaikki sille my\u00f6nnett\u00e4v\u00e4 tuet olisi lakkautettava. Yleens\u00e4 tuollaisilla toimenpiteill\u00e4 vaikutetaan - jos niill\u00e4 yleens\u00e4 saadaan lainkaan vaikutuksia aikaiseksi - diktaattoreiden sijasta t\u00e4m\u00e4n ep\u00e4varman maan tavallisten kansalaisten tilanteeseen. T\u00e4st\u00e4 asiasta olen t\u00e4ysin samaa mielt\u00e4 espanjalaisen kollegani kanssa. Vallan v\u00e4\u00e4rink\u00e4ytt\u00f6 on tuomittava ankarasti. Meid\u00e4n on my\u00f6s aloitettava vuoropuhelu presidentti Obiangin kanssa sellaisen tilanteen luomiseksi, jossa oppositio voisi hiljalleen palata toimiinsa ja maanpaossa olevat poliitikot voisivat palata koteihinsa. Lis\u00e4ksi meid\u00e4n on varmistettava, ett\u00e4 suunniteltu oikeusj\u00e4rjestelm\u00e4n uudistamista koskeva konferenssi voidaan todella j\u00e4rjest\u00e4\u00e4. Euroopassa esimerkiksi Kosovon ja Bosnian ja Hetsegovinan kehitt\u00e4minen aloitettiin yhteiskunnan rakenteista, ja n\u00e4iden maiden olosuhteet ovat nyt selke\u00e4mm\u00e4t ja vakaammat. Jos n\u00e4in ei tehd\u00e4 P\u00e4iv\u00e4ntasaajan Guineassa, tavoitteita ei saavuteta koskaan. PARSEP Emme voi olettaa, ett\u00e4 P\u00e4iv\u00e4ntasaajan Guinea voisi hetkess\u00e4 muuttua eurooppalaisten vaatimusten mukaiseksi. Se edellytt\u00e4\u00e4 vuosikausia kest\u00e4v\u00e4\u00e4 kehitt\u00e4mist\u00e4, jota ei ole edes viel\u00e4 aloitettu. Kuten jo totesin, seuraavaksi on aloitettava vuoropuhelu presidentti Obiangin kanssa ja tuettava oikeusj\u00e4rjestelm\u00e4\u00e4 koskevan konferenssin j\u00e4rjest\u00e4mist\u00e4, oppositioj\u00e4senten vapauttamista, oppositiopuolueiden laillistamista sek\u00e4 uusien vaalien j\u00e4rjest\u00e4mist\u00e4 kansainv\u00e4listen tarkkailijoiden tuella. Siin\u00e4 ovat seuraavat teht\u00e4v\u00e4mme, mutta jos nyt menemme asioiden edelle, seuraukset voivat olla tuhoisia. Siksi suosittelenkin harkitumpaa l\u00e4hestymistapaa. Pakotteiden asettaminen ei riit\u00e4 ratkaisuksi. Meid\u00e4n ei pit\u00e4isi osoittaa syytt\u00e4v\u00e4\u00e4 sormea muita kohti vaan hyv\u00e4ksy\u00e4 ensin omat vastuumme. Toisin sanoen meid\u00e4n on tarkasteltava omien \u00f6ljy-yhti\u00f6idemme asemaa t\u00e4ss\u00e4 valitettavasti eritt\u00e4in likaisessa peliss\u00e4. PARSEP Arvoisa puhemies, kollegani ja yst\u00e4v\u00e4ni j\u00e4sen Ojeda k\u00e4ytti sanaa \"alij\u00e4\u00e4m\u00e4inen\" viitatessaan P\u00e4iv\u00e4ntasaajan Guinean hallituksen demokratia- ja ihmisoikeuspolitiikkaan. PARSEP Talous- ja rahoitusasioista vastaavaa komission j\u00e4sent\u00e4 Solbesia miellytt\u00e4vin sanoin voitaisiin sanoa, ett\u00e4 Guinean hallituksen alij\u00e4\u00e4m\u00e4 n\u00e4iss\u00e4 asioissa on sit\u00e4 luokkaa, ett\u00e4 voidaan puhua vararikosta. Vararikon lis\u00e4ksi hallitus laiminly\u00f6 jatkuvasti jokaista tekem\u00e4\u00e4ns\u00e4 sitoumusta, jonka se on tehnyt ensinn\u00e4kin Euroopan unionille ja yleisesti kansainv\u00e4liselle yhteis\u00f6lle, oikeusvaltioon siirtymisest\u00e4. PARSEP \u00c4skett\u00e4iset vaalit olivat selv\u00e4 osoitus siit\u00e4, ettei presidentti Obiang halua maahansa demokratiaa tai ihmisoikeuksia. Viel\u00e4 selke\u00e4mpi osoitus t\u00e4st\u00e4 oli oppositioj\u00e4senten oikeudenk\u00e4ynti, jonka seurauksena j\u00e4senet joutuivat poliittisiksi vangeiksi ja Black Beach -vankilassa tapahtui kidutuskuolemia, jotka parlamentti on tuominnut monesti. PARSEP Euroopan unioni on osaltaan vastuussa demokratian ja ihmisoikeuksien toteutumisesta P\u00e4iv\u00e4ntasaajan Guineassa, ja joillakin sen j\u00e4senvaltioista on kaksinkertainen vastuu. Kaksi j\u00e4senvaltiota on t\u00e4ll\u00e4 hetkell\u00e4 j\u00e4senen\u00e4 turvallisuusneuvostossa: Ranska ja Espanja. T\u00e4m\u00e4n vuoksi minusta on v\u00e4ltt\u00e4m\u00e4t\u00f6nt\u00e4, ett\u00e4 Euroopan unioni - ja vaikkapa n\u00e4iden maiden ehdotuksesta - hyv\u00e4ksyy yhteisen toiminnan sitoutuakseen demokratian ja ihmisoikeuksien kunnioittamiseen ja ett\u00e4 unioni keskustelee er\u00e4\u00e4n toisen maan kanssa, joka on \u00e4\u00e4rimm\u00e4isen kiinnostunut Guineasta. Sit\u00e4 eiv\u00e4t kiinnosta niink\u00e4\u00e4n demokratia tai ihmisoikeudet, vaan \u00f6ljy. Kyse on siis Yhdysvalloista. PARSEP T\u00e4m\u00e4n vuoksi on eritt\u00e4in t\u00e4rke\u00e4\u00e4, ett\u00e4 p\u00e4\u00e4t\u00f6slauselmaan j\u00e4tet\u00e4\u00e4n kohta, jossa k\u00e4sitell\u00e4\u00e4n \u00f6ljy-yhti\u00f6ille laadittavia k\u00e4yt\u00e4nnes\u00e4\u00e4nt\u00f6j\u00e4, sill\u00e4 emme voi antaa niiden toimia ilman s\u00e4\u00e4nt\u00f6j\u00e4. \u00d6ljy, tuo mahtava voimavara, joka k\u00f6yhdytt\u00e4\u00e4 P\u00e4iv\u00e4ntasaajan Guinean kansalaisia, on nyt keskeisimm\u00e4ss\u00e4 asemassa ja presidentti Obiang Ngueman hallituksen vahvuuden tae. Euroopan unionin on siis toimittava p\u00e4\u00e4tt\u00e4v\u00e4isesti. Meid\u00e4n on hyv\u00e4ksytt\u00e4v\u00e4 t\u00e4m\u00e4 yhteinen toiminta, keskusteltava Yhdysvaltojen kanssa ja sovittava yhteisty\u00f6st\u00e4 Washingtonin p\u00e4\u00e4tt\u00e4jien kanssa. Lis\u00e4ksi meid\u00e4n on vaadittava \u00f6ljy-yhti\u00f6it\u00e4 noudattamaan velvoitteitaan, eli yksinkertaisesti t\u00e4ytt\u00e4m\u00e4\u00e4n velvollisuutensa kansaa kohtaan. PARSEP Komission on arvioitava \u00f6ljyalan tapahtumia P\u00e4iv\u00e4ntasaajan Guineassa, ja lis\u00e4ksi pyyt\u00e4isin komissiota viel\u00e4 harkitsemaan, ett\u00e4 se avaisi j\u00e4lleen edustustonsa Malabossa. Yhdysvallat aikoo avata l\u00e4hetyst\u00f6ns\u00e4 uudelleen. Miksik\u00f6? \u00d6ljyn vuoksi. Unioninkin on avattava edustustonsa uudelleen. Miksik\u00f6? Ihmisoikeuksien ja demokratian vuoksi. PARSEP Euroopan sosiaalidemokraatit antavat siis t\u00e4yden tukensa p\u00e4\u00e4t\u00f6slauselmalle, koska sit\u00e4 kannattavat kaikki ne, jotka voivat sanoa tuhansia kannattajia vaalikokouksiinsa koonneen CPDS:n presidenttiehdokkaan Celestino Bacalen tavoin, ett\u00e4 Euroopan unioni tukee P\u00e4iv\u00e4ntasaajan Guineaa samoin kuin puhemies Cox, joka aikanaan puhui selke\u00e4sti ja vahvasti Guinean demokratian ja ihmisoikeuksien puolesta. PARSEP Arvoisa puhemies, komissio on yht\u00e4 huolissaan ihmisoikeustilanteesta P\u00e4iv\u00e4ntasaajan Guineassa kuin parlamenttikin. Komission mielest\u00e4 tilanne on huonontunut merkitt\u00e4v\u00e4sti viimeisen vuoden aikana, ja vuoden 2001 loppuun asti kest\u00e4nyt jossain m\u00e4\u00e4rin my\u00f6nteinen kehitys on pys\u00e4htynyt. Tilanteen heikentyminen her\u00e4tt\u00e4\u00e4 ep\u00e4ilyksi\u00e4 siit\u00e4, pyrkiik\u00f6 presidentti Obiangin hallitus tosissaan edist\u00e4m\u00e4\u00e4n demokratiaa ja ihmisoikeuksien kunnioittamista. PARSEP Komissio my\u00f6s yhtyy parlamentin ep\u00e4ilyksiin ja arvioihin, joita se on esitt\u00e4nyt 15. joulukuuta j\u00e4rjestetyist\u00e4 vaaleista, kuten perustuslain m\u00e4\u00e4r\u00e4ysten rikkomisesta ja vaalitarkkailijoiden toiminnan est\u00e4misest\u00e4. Vaalit eiv\u00e4t selv\u00e4stik\u00e4\u00e4n olleet demokraattiset. PARSEP Presidentti Obiangin lokakuussa antaman yleisen armahduksen olisi koskettava kaikkia presidentin poliittisia vastustajia, jotka vangittiin kev\u00e4\u00e4ll\u00e4 2002 ja tuomittiin asiattomissa oikeudenk\u00e4ynneiss\u00e4. PARSEP Olemme kaikki samaa mielt\u00e4 P\u00e4iv\u00e4ntasaajan Guinean kanssa teht\u00e4v\u00e4st\u00e4 yhteisty\u00f6st\u00e4, jonka luonne riippuu vahvasti demokratisointiprosessissa ilmenevist\u00e4 ongelmista ja ihmisoikeuskysymysten kehittymisest\u00e4 maassa. Yhteisty\u00f6 oli jumissa useita vuosia, ja osittainen yhteisty\u00f6 saatiin k\u00e4ynnistetty\u00e4 viisi vuotta sitten. On totta, ett\u00e4 hein\u00e4kuussa 2000 allekirjoitetussa kahdeksannen Euroopan kehitysrahaston maaohjelmassa korostettiin demokratian, ihmisoikeuksien ja kansalaisyhteiskunnan tukemista sek\u00e4 oikeusvaltiota yhten\u00e4 ensisijaisena toiminta-alueenamme. Sen mukaan muihin toimenpiteisiin my\u00f6nnett\u00e4v\u00e4n unionin tuen edellytyksen\u00e4 on, ett\u00e4 Guinean hallitus todella toimii t\u00e4ll\u00e4 alalla. Pyrimme t\u00e4ten edist\u00e4m\u00e4\u00e4n demokratiaa ja ihmisoikeuksien kunnioittamista. PARSEP P\u00e4\u00e4t\u00f6slauselman 7 kohdasta haluaisin huomauttaa, ett\u00e4 komissio ehdotti lokakuussa P\u00e4iv\u00e4ntasaajan Guinean hallitukselle, ett\u00e4 oikeusvaltion lujittamiseen, ihmisoikeuksien edist\u00e4miseen ja kunnioittamiseen liittyvien toimien yleist\u00e4 toteutumista tarkasteltaisiin yhteisty\u00f6ohjelman mukaisesti. Jotta voimme p\u00e4\u00e4tt\u00e4\u00e4 yhteisty\u00f6toimien jatkamisesta P\u00e4iv\u00e4ntasaajan Guinean kanssa, meid\u00e4n on saatava tiet\u00e4\u00e4, mik\u00e4 on tarkastelun tilanne ja millaisia p\u00e4\u00e4telmi\u00e4 tarkastelusta voidaan tehd\u00e4. T\u00e4ll\u00f6in voimme p\u00e4\u00e4tt\u00e4\u00e4 uuden maaohjelman allekirjoittamisesta yhdeks\u00e4nnen Euroopan kehitysrahaston yhteydess\u00e4. PARSEP Lis\u00e4ksi komissio katsoo, kuten monet ovat todenneet, ett\u00e4 toimiamme on lis\u00e4tt\u00e4v\u00e4 ja kaikkien kansainv\u00e4listen instituutioiden on lis\u00e4tt\u00e4v\u00e4 painostusta. Etenkin Kansainv\u00e4lisen valuuttarahaston on lis\u00e4tt\u00e4v\u00e4 painostusta, mutta ei yksin\u00e4\u00e4n sen. N\u00e4in voimme varmistaa \u00f6ljyvarojen avoimen ja tehokkaan hallinnoinnin, jotta takaamme, ett\u00e4 n\u00e4iden voimavarojen lis\u00e4\u00e4ntyminen n\u00e4kyy vastaavasti P\u00e4iv\u00e4ntasaajan Guinean kansalaisten elintason nousuna. PARSEP Kiitos komission j\u00e4sen Solbes Mira. PARSEP Keskustelu on p\u00e4\u00e4ttynyt. PARSEP \u00c4\u00e4nestys toimitetaan iltap\u00e4iv\u00e4ll\u00e4 keskustelujen p\u00e4\u00e4tytty\u00e4. PARSEP Esityslistalla on seuraavana yhteiskeskustelu seuraavista viidest\u00e4 T?et?eniaa k\u00e4sittelev\u00e4st\u00e4 p\u00e4\u00e4t\u00f6slauselmaesityksest\u00e4: PARSEP B5-0037/2003 Paasilinna ja Sakellariou PSE-ryhm\u00e4n puolesta: EYTJ:n mandaatin jatkamatta j\u00e4tt\u00e4minen T?et?eniassa, PARSEP B5-0041/2003 Suominen, Oostlander ja Posselt PPE-DE-ryhm\u00e4n puolesta: EYTJ:n mandaatin jatkamatta j\u00e4tt\u00e4minen T?et?eniassa, PARSEP B5-0042/2003 Malmstr\u00f6m ELDR-ryhm\u00e4n puolesta: T?et?enia, PARSEP B5-0047/2003 Staes, Wuori, Schroedter, Cohn-Bendit, Isler B\u00e9guin ja Hautala Verts/ALE-ryhm\u00e4n puolesta: T?et?enian tilanne, PARSEP B5-0049/2003 Ainardi ja Vinci GUE/NGL-ryhm\u00e4n puolesta: T?et?enian tilanne. PARSEP Arvoisa puhemies, valitettavasti T?et?enian tilanne ei ole todellakaan parantunut, vaikka Ven\u00e4j\u00e4\u00e4 on taivuteltu j\u00e4rkev\u00e4\u00e4n ratkaisuun. Olemme toistuvasti painottaneet, ett\u00e4 terrorismin torjunta on t\u00e4ysin perusteltua ja todella v\u00e4ltt\u00e4m\u00e4t\u00f6nt\u00e4. T?et?enian tilanteessa - kuten vastaavissa tilanteissa Irakissa ja muilla alueilla - olemme kuitenkin aina painottaneet, ett\u00e4 pommit, sotilaalliset hy\u00f6kk\u00e4ykset ja muut sotatoimet eiv\u00e4t sovi terrorismin ongelmien ratkaisemiseen. Siksi meid\u00e4n on teht\u00e4v\u00e4 my\u00f6s Ven\u00e4j\u00e4lle selv\u00e4ksi, ett\u00e4 olemme eritt\u00e4in pettyneit\u00e4 siihen, ettei Ven\u00e4j\u00e4 - joka on ja jonka kuuluisikin olla Euroopan unionin kumppanivaltio - ole onnistunut noudattamaan j\u00e4rkev\u00e4\u00e4 toimintatapaa torjuessaan sek\u00e4 terrorismin syit\u00e4 ett\u00e4 itse terrorismia. PARSEP Olemme hyvin pettyneit\u00e4 siihen, ett\u00e4 ihmisi\u00e4 pakotetaan palaamaan tuohon turvattomaan maahan ja ett\u00e4 jopa pakolaisia kotiutetaan v\u00e4kipakolla. Meid\u00e4n mielest\u00e4mme on k\u00e4ytett\u00e4v\u00e4 kaikki keinot, jotta tilanteessa p\u00e4\u00e4st\u00e4\u00e4n rauhanomaiseen ratkaisuun. Yhten\u00e4 vaihtoehtona voisi olla kansan\u00e4\u00e4nestys, joka j\u00e4rjestet\u00e4\u00e4n puolueettomissa olosuhteissa. Ihmisill\u00e4 pit\u00e4isi olla mahdollisuus ilmaista kantansa, ja mielest\u00e4ni Putinin ja Ven\u00e4j\u00e4n maineelle tekisi hyv\u00e4\u00e4, jos Ven\u00e4j\u00e4 ottaisi j\u00e4lleen k\u00e4ytt\u00f6\u00f6n tai saisi oikeastaan ensimm\u00e4ist\u00e4 kertaa toteutettua j\u00e4rkev\u00e4n ja rauhanomaisen politiikan T?et?eniaa kohtaan. T?et?eniasta kantautuvat uutiset ovat meist\u00e4 kaikista viel\u00e4kin eritt\u00e4in p\u00f6yristytt\u00e4vi\u00e4. Ven\u00e4j\u00e4n joukot rusentavat tavalliset ihmiset taistellessaan T?et?enian tasavallan kapinallisia joukkoja vastaan. Ihmiset asuvat harvinaisen kurjissa oloissa sek\u00e4 T?et?eniassa ett\u00e4 sen ulkopuolella. Pakenemaan p\u00e4\u00e4sseet pakotetaan palaamaan alueelle, jossa on mahdotonta el\u00e4\u00e4. Sotilaiden rikoksista ei aina nosteta syytteit\u00e4. P\u00e4\u00e4t\u00f6slauselmalla haluamme protestoida kaikkea t\u00e4t\u00e4 vastaan. Grozny on edelleen \u00e4\u00e4rimm\u00e4isen turvaton kaupunki, mik\u00e4 osoittaa, ett\u00e4 niin kutsuttujen T?et?enian kapinallisten vastainen taistelu ei ole p\u00e4\u00e4ttym\u00e4\u00e4n p\u00e4ink\u00e4\u00e4n. Rakennuksia r\u00e4j\u00e4ytell\u00e4\u00e4n ja ihmiset, joiden kanssa Euroopan parlamentin valtuuskunnan piti k\u00e4yd\u00e4 avointa keskustelua, makaavat kaikki sairaalassa. Niinp\u00e4 ven\u00e4l\u00e4iset eiv\u00e4t voineet muuta kuin lyk\u00e4t\u00e4 Euroopan parlamentin valtuuskunnan vierailua T?et?eniaan. Lykk\u00e4\u00e4misen virallinen syy on ehk\u00e4 outo, mutta diplomaattisten kanavien kautta kuulimme lykk\u00e4\u00e4misen johtuneen siit\u00e4, ettei turvallisuutta voitu t\u00e4ysin taata, mik\u00e4 vaikuttaa Ven\u00e4j\u00e4n valtiosta skandaalilta. Sit\u00e4 se onkin. PARSEP Etyj:\u00e4 pidet\u00e4\u00e4n kaikkialle nen\u00e4ns\u00e4 tunkevana jokapaikanh\u00f6yl\u00e4n\u00e4, jonka mandaattia ei pit\u00e4isi jatkaa. Vastustamme my\u00f6s t\u00e4t\u00e4. Haluamme todella, ett\u00e4 Etyj:n mandaatti jatkuu, sill\u00e4 Etyj on ehdottoman v\u00e4ltt\u00e4m\u00e4t\u00f6n avuntarjoamisen ja poliittisten aloitteiden kannalta, kun my\u00f6hemmin tarkkaillaan mahdollista kansan\u00e4\u00e4nestyst\u00e4. Toinen jokapaikanh\u00f6yl\u00e4, M\u00e9decins sans Fronti\u00e8res -j\u00e4rjest\u00f6n hanketta T?et?enian naapurivaltiossa Dagestanissa johtanut Erkel on kaapattu. Tekij\u00e4\u00e4 emme tied\u00e4. Lunnaita ei ole vaadittu. On siis t\u00e4ysin mahdollista, ett\u00e4 viranomaiset ovat vanginneet miehen, kuten aiemminkin on tapahtunut. Olen kysynyt asiasta komissiolta ja neuvostolta. Useiden muiden ryhmien kollegat ovat ottaneet kysymyksen esille ja tukevat asiaa t\u00e4ysin. Haluamme, ett\u00e4 komissio toimii saadakseen selville, mit\u00e4 Erkelille tapahtui, ja katsoo, olisiko h\u00e4net mahdollista saada vapaaksi. Euroopan unionin on joka tapauksessa teht\u00e4v\u00e4 kaikki mahdollinen poliittiselta kannalta sek\u00e4 yhteis\u00f6n kannalta, mik\u00e4 vahvistetaan p\u00e4\u00e4t\u00f6slauselmassa. T?et?eenijohtajien \"valkoinen lista\", jota er\u00e4\u00e4ss\u00e4 tarkistuksessa ehdotettiin, vaikuttaa meist\u00e4 kuitenkin haitalliselta. Se on rajoittava. Ryhm\u00e4mme on taipuvainen uskomaan, ett\u00e4 Tanskan hallitus noudatti todella hyv\u00e4\u00e4 toimintamallia t\u00e4ss\u00e4 kysymyksess\u00e4. He eiv\u00e4t tarvinneet \"valkoista listaa\". PARSEP Arvoisa puhemies, t?et?eenien suhteen olemme vaarassa syyllisty\u00e4 samaan rikokseen, mik\u00e4 ilmeni eurooppalaisten toimettomuutena juutalaisten kansanmurhassa: v\u00e4linpit\u00e4m\u00e4tt\u00f6myyteen. V\u00e4linpit\u00e4m\u00e4tt\u00f6myys tekee meist\u00e4 rikoskumppaneita. T?et?eniassa on kuollut 80 000 ihmist\u00e4 vuoden 1999 j\u00e4lkeen. Satojatuhansia on joutunut pakolaisiksi. Arvostamme kyll\u00e4 Ven\u00e4j\u00e4n yhteisty\u00f6t\u00e4 terrorismin torjunnassa, mutta se ei ole mik\u00e4\u00e4n syy sulkea silmi\u00e4 molempien osapuolten julmuuksilta T?et?enian ihmisi\u00e4 kohtaan. Eteenp\u00e4in p\u00e4\u00e4st\u00e4\u00e4n vain neuvottelemalla poliittisesta ratkaisusta, kansan\u00e4\u00e4nestyksest\u00e4 ja vaaleista. Etyj:lle on annettava lupa j\u00e4\u00e4d\u00e4 avustamaan prosessissa, aivan kuten ven\u00e4l\u00e4iset ovat Kosovossa. PARSEP Ven\u00e4j\u00e4 yritt\u00e4\u00e4 saada T?et?enian presidentin l\u00e4hettil\u00e4\u00e4n, Ahmed Zakajevin luovutettavaksi. Tanskan hallitus kielt\u00e4ytyi, sill\u00e4 todisteita h\u00e4nen osallistumisestaan rikolliseen toimintaan ei esitetty. Yhdistyneen kuningaskunnan hallitus k\u00e4sittelee nyt vastaavaa pyynt\u00f6\u00e4. Sen olisi kerrottava Ven\u00e4j\u00e4lle, ett\u00e4 oikeuden p\u00e4\u00e4t\u00f6sten vastavuoroisen tunnustamisen periaatteen mukaisesti tunnustamme unionissa Tanskan p\u00e4\u00e4t\u00f6ksen ja kielt\u00e4ydymme luovuttamasta Zakajevia Ven\u00e4j\u00e4lle. Muutoin ei olisi mit\u00e4\u00e4n mielt\u00e4 pit\u00e4\u00e4 voimassa periaatetta kansallisten p\u00e4\u00e4t\u00f6sten vastavuoroisesta tunnustamisesta Euroopan unionissa. PARSEP Arvoisa puhemies, hyv\u00e4t kollegat. Uskoakseni keskustelemme nyt kolmatta tai nelj\u00e4tt\u00e4 kertaa T?et?eniasta vuoden sis\u00e4ll\u00e4, mik\u00e4 osoittaa, miten p\u00e4\u00e4tt\u00e4v\u00e4isesti parlamentti haluaa ilmaista kantansa t\u00e4st\u00e4 sodasta. Tietojen mukaan ven\u00e4l\u00e4iset yst\u00e4v\u00e4mme eiv\u00e4t ole erityisen otettuja t\u00e4llaisista keskusteluista eiv\u00e4tk\u00e4 p\u00e4\u00e4t\u00f6slauselmistamme. Minulle on aina opetettu, ett\u00e4 yst\u00e4ville on aina kerrottava totuus, ettei mit\u00e4\u00e4n saa salata. Jos yst\u00e4v\u00e4t eiv\u00e4t puhu asioista toisilleen, se on v\u00e4\u00e4rin. Haluaisin l\u00e4hett\u00e4\u00e4 t\u00e4m\u00e4n viestin ven\u00e4l\u00e4isille yst\u00e4villemme. PARSEP T?et?enian tilanne on paha, se meid\u00e4n on tunnustettava. Tied\u00e4mme kaikki, ett\u00e4 my\u00f6s Ven\u00e4j\u00e4ll\u00e4 on oltava rohkeutta tunnustaa, ett\u00e4 tilanne on paha. Monet kollegani ovat jo tuoneet t\u00e4m\u00e4n esiin. Ihmisi\u00e4 katoaa, heit\u00e4 kidutetaan ja avustusty\u00f6ntekij\u00f6it\u00e4 kaapataan. Meneill\u00e4\u00e4n on todellinen sissisota. Ven\u00e4l\u00e4iset k\u00e4yv\u00e4t likaista sotaa, mutta niin k\u00e4yv\u00e4t t?et?eenikapinallisetkin. Ingu?ian pakolaisleirien tilanne on kauhistuttava. Ven\u00e4j\u00e4 yritt\u00e4\u00e4 nyt silti pakottaa n\u00e4m\u00e4 ihmiset palaamaan T?et?eniaan. Tied\u00e4mme kaikki, ettei palaaminen ole viisasta talvikuukausien aikana, eik\u00e4 se itse asiassa ole mahdollistakaan. Etyj halutaan pois T?et?eniasta. Ven\u00e4j\u00e4 ei selv\u00e4stik\u00e4\u00e4n halua sinne tunkeilevia jokapaikanh\u00f6yli\u00e4. Samoin Ven\u00e4j\u00e4 est\u00e4\u00e4 nyt meid\u00e4n valtuuskuntaamme matkustamasta. Se k\u00e4ytt\u00e4\u00e4 mielest\u00e4ni n\u00e4enn\u00e4isesti p\u00e4tev\u00e4lt\u00e4 vaikuttavaa perustetta puhemiehelle osoitetussa kirjeess\u00e4. On totta, ettei Ven\u00e4j\u00e4 hallitse T?et?enian tilannetta t\u00e4ll\u00e4 hetkell\u00e4 eik\u00e4 pysty takaamaan turvallisuuttamme. PARSEP Tied\u00e4mme kaikki, ett\u00e4 ven\u00e4l\u00e4iset kollegamme ovat hyvin tyytym\u00e4tt\u00f6mi\u00e4 antamiimme p\u00e4\u00e4t\u00f6slauselmiin. T\u00e4m\u00e4 p\u00e4\u00e4t\u00f6slauselma on kuitenkin hyvin tasapuolinen. P\u00e4\u00e4t\u00f6slauselma osoittaa, ett\u00e4 Euroopan parlamentti haluaa ratkaisun syntyv\u00e4n ja ett\u00e4 se haluaa todella auttaa asiassa. Kehotankin siksi ven\u00e4l\u00e4isi\u00e4 kollegojamme jatkamaan vuoropuhelua Euroopan parlamentin kanssa, jotta v\u00e4lillemme syntyy vilpit\u00f6n ja asiallinen yhteys ja jotta varmistetaan, ett\u00e4 osapuolet saadaan lopulta tuotua yhteen poliittisen ratkaisun l\u00f6yt\u00e4miseksi. PARSEP Arvoisa puhemies, saanen t\u00e4m\u00e4np\u00e4iv\u00e4isen keskustelun alkajaisiksi ilmaista my\u00f6t\u00e4tuntoni Groznyn terrorihy\u00f6kk\u00e4yksen uhrien sukulaisille. Terrorismista huokuu mit\u00e4 syvin halveksunta ihmisel\u00e4m\u00e4\u00e4 kohtaan, eik\u00e4 teoille l\u00f6ydy koskaan perusteita. Syylliset on saatava oikeuden eteen. PARSEP Vetoan kuitenkin k\u00e4rsim\u00e4\u00e4n joutuneisiin sek\u00e4 Ven\u00e4j\u00e4n hallitukseen, etteiv\u00e4t he vastaisi samalla mitalla, vaan pyrkisiv\u00e4t rauhanomaiseen ratkaisuun. Siviilihallintoon kohdistunut hy\u00f6kk\u00e4ys osoittaa selv\u00e4sti, ettei terroristeja kiinnosta konfliktin p\u00e4\u00e4tt\u00e4minen. Heid\u00e4n tavoitteenaan on p\u00e4invastoin pysyv\u00e4 ep\u00e4vakaisuus alueella. PARSEP Olen tyytyv\u00e4inen suunnitelmaan kansan\u00e4\u00e4nestyksen j\u00e4rjest\u00e4misest\u00e4 T?et?eniassa maaliskuussa, mik\u00e4 her\u00e4tt\u00e4\u00e4 toiveita demokratiasta ja itsehallinnosta. Ingu?ian pakolaisten rohkaiseminen palaamaan kotiin T?et?eniaan on oikea l\u00e4hestymistapa, mutta on eritt\u00e4in v\u00e4\u00e4rin k\u00e4ytt\u00e4\u00e4 pakkokeinoja. Ensin on luotava riitt\u00e4v\u00e4t olosuhteet, jotta pakolaiset voivat palata. Ihmisten on tunnettava olonsa turvalliseksi, ja he tarvitsevat asunnon, s\u00e4hk\u00f6\u00e4 ja ruokaa. Koulut on avattava ja terveydenhuolto varmistettava. Euroopan unioni on halukas toimimaan asiassa ja tarjoamaan humanitaarista apua. PARSEP Ven\u00e4j\u00e4n hallituksen p\u00e4\u00e4t\u00f6st\u00e4 olla jatkamatta Etyj:n mandaattia on siis entist\u00e4kin mahdottomampi ymm\u00e4rt\u00e4\u00e4. Mit\u00e4 poliittista vastustusta her\u00e4tt\u00e4v\u00e4t ihmisoikeuksien kunnioittamisen valvominen, pakolaisten avustaminen ja konfliktin ratkaisun tukeminen? Mielest\u00e4ni eiv\u00e4t mit\u00e4\u00e4n, ellei toimia pidet\u00e4 pikemminkin valtion sis\u00e4isiin asioihin puuttumisena kuin tarjouksena auttaa eritt\u00e4in vaikeassa tilanteessa. Ehk\u00e4p\u00e4 Ven\u00e4j\u00e4n hallitus katsoo, ett\u00e4 avun hyv\u00e4ksyminen olisi heikkouden merkki, mihin Ven\u00e4j\u00e4ll\u00e4 ei ole varaa poliittisista syist\u00e4. T\u00e4m\u00e4 lienee mielest\u00e4ni t\u00e4ydellinen v\u00e4\u00e4rink\u00e4sitys, joka voidaan kuitenkin korjata. PARSEP Euroopan parlamentin T?et?eniaan matkustavan tilap\u00e4isen valiokunnan j\u00e4senen\u00e4 olen pettynyt siihen, ett\u00e4 duuman vahvistama valiokunnan vierailu peruttiin. Olen pettynyt, koska halusimme tietenkin hankkia oman n\u00e4kemyksen tilanteesta paikan p\u00e4\u00e4ll\u00e4 - mik\u00e4 n\u00e4htiin mahdollisuutena lis\u00e4t\u00e4 keskin\u00e4ist\u00e4 ymm\u00e4rryst\u00e4 - ja ennen kaikkea siksi, koska esiin nousee poliittinen peruskysymys, johon on vastattu kielteisesti. Meille kerrottiin, ett\u00e4 vierailun lykk\u00e4\u00e4misen syyn\u00e4 on Ahmed Zakajevin vapauttaminen vankeudesta Euroopan maassa ja parlamentin hyv\u00e4ksym\u00e4 p\u00e4\u00e4t\u00f6slauselma asiasta. PARSEP Hyv\u00e4ksyn sen, ett\u00e4 duumalla on erilainen poliittinen kanta kuin meill\u00e4. Minulla on kuitenkin perustavanlaatuinen kysymys: miten voidaan toivoa rauhanomaista ratkaisua eritt\u00e4in monimutkaisiin kysymyksiin T?et?eniassa neuvottelemalla todellisten vastustajien kanssa, jos on mahdotonta hyv\u00e4ksy\u00e4, ett\u00e4 yst\u00e4v\u00e4llismielisell\u00e4 toimielimell\u00e4 voi olla erilainen poliittinen kanta? PARSEP J\u00e4sen Ludford, meid\u00e4n on ehdottomasti varottava vertaamasta kansallissosialismia ja Saksan kansallissosialistien juutalaisvainoja muihin tilanteisiin. Vertaamalla v\u00e4h\u00e4tell\u00e4\u00e4n noita hirvitt\u00e4vi\u00e4 valtiojohtoisia ihmisoikeusrikkomuksia, jotka ovat ainutlaatuisia historiassa. \u00c4lk\u00e4\u00e4 en\u00e4\u00e4 tehk\u00f6 moista vertausta. PARSEP Arvoisa puhemies, T?et?eniassa yritet\u00e4\u00e4n kolmatta kertaa toteuttaa j\u00e4rjestelm\u00e4llist\u00e4 kansanmurhaa. Toteuttajana on siirtomaavalta Ven\u00e4j\u00e4, jonka tarkoituksena on turvata strategiset edut ja voimavarat. Kansanmurha on verhoiltu terrorismin torjunnaksi, ja vaikka terrorismia esiintyy alueella, verhoilu on vain kyynist\u00e4 todellisuuden v\u00e4\u00e4ristely\u00e4. Onkin t\u00e4rke\u00e4\u00e4, ett\u00e4 sota-alueelle olisi esteet\u00f6n p\u00e4\u00e4sy, jotta tosiasiat saadaan esitetty\u00e4. PARSEP Nykytilanteessa Ven\u00e4j\u00e4 tekee kaikkensa varmistaakseen, ett\u00e4 sota liukuu n\u00e4kyvist\u00e4 hiljaisuuden muurin suojiin. Ensinn\u00e4kin komissio on ollut t\u00e4h\u00e4n asti haluton vierailemaan alueella. Olemme keskustelleet asiasta moneen kertaan komission j\u00e4senen Nielsonin kanssa, ja olemme kiirehtineet h\u00e4nt\u00e4 vierailemaan alueella. Ehk\u00e4p\u00e4 vaikutusvaltaa on jo hieman k\u00e4ytetty kulissien takana. Toiseksi Euroopan parlamentti ilmoitti haluavansa vierailla alueella. Parlamentin valtuuskuntaa estettiin matkustamasta kriisialueelle lukuisista i\u00e4nikuisista syist\u00e4. T\u00e4m\u00e4 oli valitettavaa, ja se n\u00e4kyy vienos\u00e4vyisen\u00e4 arvosteluna, joka on sis\u00e4llytetty p\u00e4\u00e4t\u00f6slauselmaan. Ennen kaikkea se oli kuitenkin skandaali! Euroopan unionin ja Ven\u00e4j\u00e4n oletetaan olevan kumppaneita, mik\u00e4 tarkoittaa, ett\u00e4 kuten kumppanillamme Ven\u00e4j\u00e4ll\u00e4 on oikeus ja mahdollisuus matkustaa kaikkialle unioniin hankkiakseen ensik\u00e4den tietoa, meill\u00e4 on oletettavasti sama vapaa ja esteet\u00f6n p\u00e4\u00e4sy Ven\u00e4j\u00e4n valta-alueelle milloin tahansa kaikkien, muiden muassa Ven\u00e4j\u00e4n Etyj:ss\u00e4 ja Euroopan neuvostossa allekirjoittamien kansainv\u00e4listen sopimuksen mukaisesti. PARSEP Kolmas yritys rakentaa hiljaisuuden muuri T?et?enian ymp\u00e4rille oli Etyj:n mandaatin lakkauttaminen. Paikalta h\u00e4\u00e4det\u00e4\u00e4n kansainv\u00e4linen j\u00e4rjest\u00f6, joka on tehnyt suuren ty\u00f6n aikaisempien T?et?enian sotien p\u00e4\u00e4tt\u00e4miseksi neuvoteltuun sovintoon. Etyj:n valvonnan alla valittiin presidentti Maskhadov, joka on ainoa t\u00e4h\u00e4n menness\u00e4 demokraattisesti valittu edustaja. H\u00e4nt\u00e4 ei siis voi sulkea vuoropuhelun ulkopuolelle. My\u00f6s kunnioitetun Saksan entisen ministerin Norbert Bl\u00fcmin johtama Saksan valtuuskunta poistettiin maasta. PARSEP Arvoisa puhemies, mielest\u00e4ni t\u00e4m\u00e4 on ehdottomasti p\u00f6yristytt\u00e4v\u00e4\u00e4. Mielest\u00e4ni meid\u00e4n on k\u00e4ytett\u00e4v\u00e4 entist\u00e4 selvemp\u00e4\u00e4 kielt\u00e4: Ven\u00e4j\u00e4lle on teht\u00e4v\u00e4 selv\u00e4ksi, ettei t\u00e4m\u00e4 voi jatkua. Haluamme n\u00e4hd\u00e4 valoa pimeydess\u00e4, haluamme lopettaa t\u00e4m\u00e4n sodan. Arvoisa puhemies, hyv\u00e4t kollegat. Mielest\u00e4ni T?et?enian nykyinen tilanne ei ole eritt\u00e4in j\u00e4nnittynyt, kuten kompromissitekstiss\u00e4 todetaan. Mielest\u00e4ni tilanne on tosiasiassa \u00e4\u00e4rimm\u00e4isen j\u00e4nnittynyt. Yht\u00e4\u00e4lt\u00e4 on useita t?et?eenisukupolvia, jotka ovat n\u00e4hneet t\u00e4h\u00e4n menness\u00e4 ainoastaan sotaa ja v\u00e4kivaltaa ja joita \u00e4\u00e4riryhm\u00e4t voivat johtaa hetken\u00e4 min\u00e4 hyv\u00e4ns\u00e4 ajattelemattomiin terroritekoihin. Toisaalta k\u00e4ynniss\u00e4 on todellinen siirtomaasota, jota johtaa Ven\u00e4j\u00e4n armeija ja johon sis\u00e4ltyy valtavan laajoja ja j\u00e4rjestelm\u00e4llisi\u00e4 etsint\u00f6j\u00e4, ihmisoikeusrikkomuksia, salamurhia ja ihmisry\u00f6st\u00f6j\u00e4. Kaiken huipuksi eversti Juri Budanov, jota syytettiin 18-vuotiaan t?et?eenityt\u00f6n raiskauksesta ja murhasta, vapautettiin h\u00e4pe\u00e4llisesti syyt\u00f6ksist\u00e4 31. joulukuuta 2002. N\u00e4in ollen on melko p\u00f6yristytt\u00e4v\u00e4\u00e4, ett\u00e4 avustusty\u00f6ntekij\u00e4\u00e4 olisi edelleen pidett\u00e4v\u00e4 vangittuna. PARSEP Mielest\u00e4ni kaksinaismoraalia olisi v\u00e4ltett\u00e4v\u00e4. Jos otamme vastaan presidentti Putinin, meid\u00e4n on teht\u00e4v\u00e4 jotain terveysministerille, joka palasi Bakuun ja jonka viisumi otettiin pois, sek\u00e4 Ahmed Zakajeville, jonka kanssa puhuin eilen puhelimessa ja joka on edelleen kotiarestissa Yhdistyneess\u00e4 kuningaskunnassa. Olisi teht\u00e4v\u00e4 jotain my\u00f6s ulkoministerille, jota pid\u00e4tell\u00e4\u00e4n viisumitta Yhdysvalloissa. My\u00f6s n\u00e4ille henkil\u00f6ille on my\u00f6nnett\u00e4v\u00e4 viisumit, sill\u00e4 rauhan solmimiseen tarvitaan molemmat osapuolet. PARSEP Arvoisa puhemies, valitettavasti olemme osittain vastuussa muualla maailmassa tapahtuneista julmuuksista ja kansanmurhista, sill\u00e4 Euroopan unioni ei koskaan uskalla vakavissaan astua Ven\u00e4j\u00e4n, Kiinan, Iranin tai Israelin kaltaisten suurvaltojen varpaille. Pit\u00e4isi olla itsest\u00e4\u00e4n selv\u00e4\u00e4, ett\u00e4 Etyj:n mandaattia olisi jatkettava, ett\u00e4 kansainv\u00e4listen edustajien olisi p\u00e4\u00e4st\u00e4v\u00e4 T?et?eniaan ja ett\u00e4 T?et?eniassa vakaviin rikoksiin syyllistyneit\u00e4 olisi rangaistava. Tanskalaisena haluaisin eritoten kehottaa kahden tarkistuksen hyv\u00e4ksymist\u00e4. Euroopan unioni ei r\u00e4p\u00e4ytt\u00e4nyt silm\u00e4\u00e4ns\u00e4k\u00e4\u00e4n, kun Ven\u00e4j\u00e4 asetti pakotteita Tanskalle sen j\u00e4lkeen, kun Tanska oli perustellusti kielt\u00e4ytynyt luovuttamasta Zakajevia terroristina, mik\u00e4 oli ainoa sovelias ratkaisu. T\u00e4llaista tilannetta eiv\u00e4t saisi tulevaisuudessa aiheuttaa sen enemp\u00e4\u00e4 t?et?eenipoliitikot kuin j\u00e4senvaltiotkaan. PARSEP Arvoisa puhemies, hyv\u00e4t kollegat. Mielest\u00e4ni teksti, jota valmistelemme \u00e4\u00e4nestyst\u00e4 varten, on yksinkertaisesti t\u00e4ysin vastenmielinen ja halveksittava. J\u00e4sen Oostlander ei ole t\u00e4ll\u00e4 hetkell\u00e4 l\u00e4sn\u00e4, mutta kollegamme, jotka ovat kahden vuoden ajan jarrutelleet parlamentin pyynt\u00f6j\u00e4 saada Milosevic syytteeseen, ovat samoja kollegoja, jotka nyt toteuttavat muka realistista politiikkaa Ven\u00e4j\u00e4n federaatiota ja presidentti Putinia kohtaan ja jotka jokaisen tilaisuuden tullen peittelev\u00e4t T?et?enian tosiasiallista tilannetta, todellista kansanmurhaa. PARSEP Ilmoitamme olevamme tyytym\u00e4tt\u00f6mi\u00e4 Budanovin vapauttamiseen. Mielest\u00e4ni meid\u00e4n pit\u00e4isi olla v\u00e4hint\u00e4\u00e4nkin \u00e4\u00e4rimm\u00e4isen j\u00e4rkyttyneit\u00e4. Pahoittelemme sit\u00e4, ettemme voi matkustaa T?et?eniaan. Vierailua on nyt lyk\u00e4tty nelj\u00e4 kertaa, ja kaksi vuotta on kulunut siit\u00e4, kun Euroopan parlamentin valtuuskunnan piti matkustaa T?et?eniaan. PARSEP Teemme virheit\u00e4 ja hukkaamme n\u00e4kyvist\u00e4 kaikki kiintopisteet. Unohdamme, ett\u00e4 vuosina 1940-1945 maissamme oli vastarintaa, joka kohdistui miehitt\u00e4j\u00e4vallan symboleihin: Vichy\u00e4, Quislingi\u00e4 ja kaikkia miehitt\u00e4j\u00e4vallan kanssa yhteisty\u00f6t\u00e4 tehneit\u00e4 vastaan. Nyt vastarintaa on hy\u00f6kk\u00e4ys hallinnon Groznyn p\u00e4\u00e4majaan, mik\u00e4 on k\u00e4sitt\u00e4m\u00e4t\u00f6n teko minun kaltaisestani v\u00e4kivaltaa karttavasta henkil\u00f6st\u00e4. T?et?eenit haluavat syd\u00e4mens\u00e4 pohjasta tehd\u00e4 yhteisty\u00f6t\u00e4 Ven\u00e4j\u00e4n kanssa, mutta kuitenkin syyt\u00e4mme heit\u00e4 terrorihy\u00f6kk\u00e4yksist\u00e4. N\u00e4m\u00e4 eiv\u00e4t ole terrorihy\u00f6kk\u00e4yksi\u00e4. Hy\u00f6kk\u00e4ys kohdistui FSB:n ty\u00f6ntekij\u00f6ihin, jotka olivat t?et?eenitaistelijoiden eli vastarinnan strateginen kohde. Olemme pudonneet k\u00e4rryilt\u00e4 t\u00e4ysin. Yhdist\u00e4mme kaiken terrorismiin. PARSEP T\u00e4m\u00e4 on eritt\u00e4in vakavaa, ja on my\u00f6s eritt\u00e4in vakavaa puoltaa yht\u00e4\u00e4lt\u00e4 poliittista ratkaisua ja tukea toisaalta miehitt\u00e4j\u00e4vallan j\u00e4rjest\u00e4m\u00e4\u00e4 kansan\u00e4\u00e4nestyst\u00e4; puoltaa poliittisia neuvotteluja ja kielt\u00e4\u00e4 samalla k\u00e4yt\u00e4nn\u00f6ss\u00e4 T?et?enian parlamentin j\u00e4senilt\u00e4 ja ministereilt\u00e4 p\u00e4\u00e4sy alueellemme tapaamaan omia poliitikkojamme Euroopan parlamentissa, Euroopan komissiossa ja Eurooppa-neuvostossa sill\u00e4 tekosyyll\u00e4, ett\u00e4 se olisi mahdotonta. Tied\u00e4mme kaikki vallan hyvin, ett\u00e4 14 artiklan 3 kohdan nojalla pystyimme laatimaan mustia listoja Jugoslavian ja Valko-Ven\u00e4j\u00e4n poliitikoista ja ett\u00e4 voimme laatia \"valkoisen listan\" t?et?eenipoliitikoista. Jos haluamme politikoida, se on teht\u00e4v\u00e4 poliitikkojen kanssa. T\u00e4lt\u00e4 kannalta katsottuna meist\u00e4 alkaa todella v\u00e4hitellen tulla osasyyllisi\u00e4 kansanmurhaan. Parlamentin on aika her\u00e4t\u00e4. Muutoin toistuvat samat sanoin kuvaamattomat toimet, joita tapahtui Bosniassa, Kroatiassa ja Kosovossa. Mielest\u00e4ni niist\u00e4 toimista on jo saatu maksaa tarpeeksi. PARSEP Arvoisa puhemies, T?et?enian konfliktista keskusteltiin yksityiskohtaisesti 11. syyskuuta edellisess\u00e4 Euroopan unionin ja Ven\u00e4j\u00e4n federaation v\u00e4lisess\u00e4 huippukokouksessa. PARSEP Valitettavasti rauhanponnistelujen jumiutuminen pakottaa meid\u00e4t j\u00e4lleen kerran kiinnitt\u00e4m\u00e4\u00e4n huomiota traagiseen tilanteeseen, joka tuhoaa T?et?eniaa. Olemme kaikki tietoisia siit\u00e4, ett\u00e4 turvallisuuden kannalta tilanne on pahentunut viimeksi Brysseliss\u00e4 pidetyn huippukokouksen j\u00e4lkeen. Haluaisin k\u00e4ytt\u00e4\u00e4 tilaisuuden hyv\u00e4kseni ja vahvistaa, ett\u00e4 unioni tuomitsee vankasti viimeaikaiset terrorihy\u00f6kk\u00e4ykset, joissa kuoli yli sata ihmist\u00e4 ja loukkaantui vakavasti paljon useampia. PARSEP Euroopan unioni vastustaa terrorismia ponnekkaasti huolimatta siit\u00e4, kuka vastaa teoista ja mitk\u00e4 ovat motiivit. Autopommit, mielivaltaiset murhat, ihmisry\u00f6st\u00f6t ja muut terroristien k\u00e4ytt\u00e4m\u00e4t menetelm\u00e4t eiv\u00e4t ole miss\u00e4\u00e4n tapauksessa perusteltuja, vaikuttivatpa ne miten oikeutetuilta hyv\u00e4ns\u00e4. PARSEP Sama p\u00e4tee luonnollisesti t\u00e4ysin T?et?eniaan, mutta niin p\u00e4tee my\u00f6s vakaumuksemme, jonka mukaan terrorismia ei voi torjua ihmisoikeuksien noudattamisen kustannuksella. PARSEP Sallikaa minun luoda ensin katsaus erityisesti ihmisoikeuksiin. Tunnustetussa asemassa olevat kansalaisj\u00e4rjest\u00f6t raportoivat jatkuvasti Ven\u00e4j\u00e4n federaation turvallisuusjoukkojen tekemist\u00e4 vakavista ihmisoikeusrikkomuksista T?et?eniassa. Murhat, kidutus, katoamiset ja muut perusihmisoikeuksien loukkaukset olisi siksi selv\u00e4sti ja ponnekkaasti tuomittava Euroopan unionissa. PARSEP Yht\u00e4 vakavaa on viranomaisten selv\u00e4 haluttomuus puuttua rikkomuksiin. Eversti Budanovia vastaan k\u00e4ydyn oikeudenk\u00e4ynnin tulos ei ole my\u00f6nteinen merkki vaan p\u00e4invastoin. Toinen kielteinen tapahtuma oli t?et?eenipakolaisten karkottaminen Aki Yurtin leirilt\u00e4 juuri ennen joulua. Euroopan unioni totesi tapahtuneen rikkovan kansainv\u00e4list\u00e4 humanitaarista periaatetta, jonka mukaan siirtym\u00e4\u00e4n joutuneet henkil\u00f6t voidaan palauttaa kotipaikkaansa ainoastaan heid\u00e4n omasta vapaasta tahdostaan ilman painostusta, jota tilanteessa k\u00e4ytettiin. PARSEP Euroopan unioni toimi pikaisesti ja vaati, ettei sama saisi toistua muilla vastaavilla leireill\u00e4, sill\u00e4 se aiheuttaisi vain k\u00e4rsimyst\u00e4 siviiliv\u00e4est\u00f6lle. Ihmiset joutuvat siirtym\u00e4\u00e4n sodan ja terrorismin v\u00e4kivallan tielt\u00e4, ja ennen kuin v\u00e4kivalta loppuu, he eiv\u00e4t voi palata kotiin henke\u00e4\u00e4n vaarantamatta. Ihmisi\u00e4 varten olisi my\u00f6s oltava olemassa tarvittava talouden infrastruktuuri, jota ei nykyisin ole, jotta he voisivat jatkaa el\u00e4m\u00e4\u00e4ns\u00e4. Haluan vakuuttaa teille, ett\u00e4 Euroopan komissio seuraa tapahtumia tarkkaan ja ryhtyy tarvittaviin toimiin, jotta tilanne ei en\u00e4\u00e4 toistu. PARSEP Humanitaarisen avun toimisto suunnittelee itse asiassa tarkkailuvaltuuskunnan l\u00e4hett\u00e4mist\u00e4 Ingu?iaan kuun lopussa. Euroopan komissio on tietoinen parlamentin enemmist\u00f6n kannasta sek\u00e4 parlamentin yrityksist\u00e4 saavuttaa yhteiset tavoitteet, jotka mainitsin juuri. Niinp\u00e4 komissio haluaa ja toivoo, ett\u00e4 suhteista duumaan vastaavan parlamentin valtuuskunnan vierailu T?et?eniaan, jota j\u00e4lleen kerran lyk\u00e4ttiin, toteutuu pian. Sen avulla Euroopan unionin kanta T?et?enian tilanteeseen tulee taatusti kuulluksi. PARSEP Toinen hyvin t\u00e4rke\u00e4 seikka on Euroopan unionin huomattava humanitaarinen apu. Euroopan unionin toistuvista huomautuksista huolimatta Ven\u00e4j\u00e4n viranomaiset ly\u00f6v\u00e4t jatkuvasti laimin tarvittavan yhteisty\u00f6n, jotta apua jakavat kansalaisj\u00e4rjest\u00f6t voisivat toimittaa avun perille turvallisissa olosuhteissa. Avustusty\u00f6ntekij\u00e4t kohtaavat edelleen lukemattomia hallinnollisia esteit\u00e4, jotka est\u00e4v\u00e4t heit\u00e4 tekem\u00e4st\u00e4 ty\u00f6t\u00e4\u00e4n tehokkaasti. Viel\u00e4 vakavampaa on, ett\u00e4 byrokratian esteet vaarantavat avustusty\u00f6ntekij\u00f6iden hengen ja turvallisuuden. PARSEP T\u00e4ss\u00e4 yhteydess\u00e4 haluaisin tuoda esiin, ett\u00e4 Arjan Erkel, M\u00e9decins sans Fronti\u00e8res -j\u00e4rjest\u00f6n ty\u00f6ntekij\u00e4, kaapattiin kuukausia sitten. Olemme vaatineet - ja toistamme vaatimuksen - ett\u00e4 Ven\u00e4j\u00e4n viranomaisten on teht\u00e4v\u00e4 kaikkensa varmistaakseen, ett\u00e4 Erkel vapautetaan v\u00e4litt\u00f6m\u00e4sti. PARSEP Jos sallitte, haluaisin nyt puhua p\u00e4\u00e4asiasta, k\u00e4sitelt\u00e4v\u00e4n\u00e4 olevasta kiireellisest\u00e4 p\u00e4\u00e4t\u00f6slauselmasta. PARSEP Kuten tied\u00e4tte, Etyj:n T?et?enian avustusryhm\u00e4n mandaatti p\u00e4\u00e4ttyi 31. joulukuuta. Mandaatin sis\u00e4ll\u00f6st\u00e4 ei p\u00e4\u00e4sty sovintoon tulevaksi vuodeksi, mik\u00e4 tarkoittaa, ett\u00e4 ryhm\u00e4n on nyt pakko tehd\u00e4 p\u00e4\u00e4t\u00f6s T?et?eniasta l\u00e4htemisest\u00e4. Ven\u00e4j\u00e4n federaatio on toistaiseksi pysynyt kannassaan, jonka mukaan T?et?enian avustusryhm\u00e4n mandaattia olisi kavennettava siten, ett\u00e4 poliittinen toiminta j\u00e4tet\u00e4\u00e4n mandaatin ulkopuolelle. T\u00e4m\u00e4 on ristiriitaista, kun otetaan huomioon, ett\u00e4 Etyj on luonteeltaan eritt\u00e4in poliittinen Euroopan turvallisuusj\u00e4rjest\u00f6. Komission mielest\u00e4 ryhm\u00e4n s\u00e4ilytt\u00e4minen on paitsi soveliasta, my\u00f6s kaikkien edun mukaista erityisesti Ven\u00e4j\u00e4n kannalta. Sama p\u00e4tee Euroopan neuvoston toimielinten ty\u00f6h\u00f6n, jossa tulee eteen entist\u00e4 enemm\u00e4n ongelmia. PARSEP N\u00e4iden kansainv\u00e4listen j\u00e4rjest\u00f6jen toiminta n\u00e4ytt\u00e4\u00e4 viel\u00e4 v\u00e4ltt\u00e4m\u00e4tt\u00f6m\u00e4mm\u00e4lt\u00e4 nyt, kun Ven\u00e4j\u00e4n federaatiolla on vakaa aikomus ryhty\u00e4 toimiin T?et?enian poliittisen tilanteen vakauttamiseksi. J\u00e4rjest\u00f6jen l\u00e4sn\u00e4olo ja apu on ainoa keino toteuttaa vakauttaminen siten, ett\u00e4 sill\u00e4 on jonkinasteinen demokraattinen oikeutus. PARSEP Wieniss\u00e4 k\u00e4yd\u00e4\u00e4n parhaillaan tiiviit\u00e4 neuvotteluja, joiden tavoitteena on varmistaa, ett\u00e4 avustusryhm\u00e4 pysyy T?et?eniassa. Asiasta keskustellaan t\u00e4n\u00e4\u00e4n my\u00f6s Ateenassa Euroopan unionin ja Ven\u00e4j\u00e4n poliittisten johtajien kesken. Toivottavasti saavutamme tavoitteen n\u00e4iden diplomatian keinojen avulla. PARSEP Kiitokset komission j\u00e4senelle. PARSEP Yhteiskeskustelu on p\u00e4\u00e4ttynyt. PARSEP Seuraavaksi toimitetaan \u00e4\u00e4nestykset. PARSEP Esityslistalla on seuraavana \u00e4\u00e4nestys lopullisesta mietinn\u00f6st\u00e4. PARSEP Esittelij\u00e4 on tehnyt t\u00e4rkeit\u00e4 huomautuksia kalastusalan yhteiskunnallisista, taloudellisista ja ymp\u00e4rist\u00f6\u00f6n liittyvist\u00e4 n\u00e4k\u00f6kohdista, jotka koskevat kalastusta kansainv\u00e4lisill\u00e4 vesill\u00e4. Mielest\u00e4ni on kuitenkin ratkaisevaa, ett\u00e4 sek\u00e4 Euroopan komissio ett\u00e4 Euroopan parlamentti pohtivat kalastuspolitiikkaa uudistettaessa kysymyst\u00e4, miten voidaan toimia Johannesburgissa vuonna 2002 pidetyss\u00e4 kest\u00e4v\u00e4n kehityksen konferenssissa sovitun toimintasuunnitelman mukaisesti, miten voidaan poistaa kaikki tuet laittomalta, kest\u00e4m\u00e4tt\u00f6m\u00e4lt\u00e4 ja s\u00e4\u00e4ntelem\u00e4tt\u00f6m\u00e4lt\u00e4 kalastukselta ja miten palautetaan uhanalaiset kalakannat vuoteen 2015 menness\u00e4. PARSEP Keskustelun kohteena oleva mietint\u00f6 ansaitsee t\u00e4yden huomion, koska siin\u00e4 esitet\u00e4\u00e4n k\u00e4yt\u00e4nn\u00f6n ratkaisu kalastajia vaivaamaan ongelmaan. Sopimusten solmiminen kolmansien maiden kanssa tarjoaa unionin k\u00e4ytt\u00f6\u00f6n lis\u00e4\u00e4 kalavaroja. Sopimusten kautta unioni varmistaa, ett\u00e4 sen kalastajilla on ty\u00f6t\u00e4, ratkaisee kalatuotteiden tarjontaan liittyv\u00e4n ongelman ja toteuttaa sosiaalis-taloudelliset teht\u00e4v\u00e4ns\u00e4 kehitysmaita kohtaan. Kannatan siksi mietinn\u00f6ss\u00e4 esitetty\u00e4: tavoitteena on lis\u00e4t\u00e4 unionin osallistumista alueellisiin kalastusj\u00e4rjest\u00f6ihin, solmia kahdenv\u00e4lisi\u00e4 sopimuksia ja uusia nykyiset sopimukset. Kalastuspolitiikan ulkoisia toimia on toteutettava j\u00e4rjestelm\u00e4llisell\u00e4 ja kest\u00e4v\u00e4ll\u00e4 tavalla. Pyyt\u00e4isin komissiota esitt\u00e4m\u00e4\u00e4n poliittisen strategian, jossa korostetaan YKP:n kansainv\u00e4list\u00e4 ulottuvuutta, jota ei voi j\u00e4tt\u00e4\u00e4 huomiotta Euroopan unionin ulkopolitiikassa (YUTP) eik\u00e4 kansainv\u00e4lisess\u00e4 kauppapolitiikassa. PARSEP Lopuksi haluan k\u00e4ytt\u00e4\u00e4 tilaisuuden hyv\u00e4kseni ja ilmaista pahoitteluni kiistoista, jotka syntyiv\u00e4t \u00e4skett\u00e4in Algarvessa espanjalaisten kalastajien ja Portugalin meriviranomaisten kesken. Portugalin meripoliisi yhytti espanjalaiset alukset, jotka harjoittivat laitonta simpukanpyynti\u00e4 Portugalin vesill\u00e4. Takaa-ajosta seurasi fyysinen k\u00e4rh\u00e4m\u00e4, jossa yksi viranomainen loukkaantui, mink\u00e4 vuoksi Portugalin laivaston korvetti joutui puuttumaan asiaan. Lis\u00e4isin viel\u00e4, ett\u00e4 espanjalaiset lainrikkojat saatiin kiinni imumenetelm\u00e4n k\u00e4yt\u00f6st\u00e4, joka on t\u00e4ysin kielletty\u00e4 Euroopan unionissa. Vaikka Portugali on varoittanut Espanjan viranomaisia, Portugalin hallitus ei ole saanut vastausta, mik\u00e4 ei kuitenkaan haittaa maiden v\u00e4lisi\u00e4 hyvi\u00e4 suhteita. PARSEP Yhteis\u00f6n laivaston toiminnan tarkoituksena unionin ulkopuolisilla kalastusalueilla on huolehtia ett\u00e4 yhteis\u00f6n markkinoilla on riitt\u00e4v\u00e4sti kalatuotteita, joista on pulaa. Lis\u00e4ksi tavoitteena on yll\u00e4pit\u00e4\u00e4 toimintaa kalastuksesta riippuvaisilla seuduilla, jotka ovat perinteisesti olleet ep\u00e4suotuisia ja vailla taloudellisen toiminnan vaihtoehtoja. PARSEP Vaikka j\u00e4sen Migu\u00e9lez Ramosin laatima mietint\u00f6 on valiokunta-aloitteinen, siihen kiinnitet\u00e4\u00e4n toivottavasti asianmukaista huomiota komission esittelyss\u00e4, joka toivoaksemme j\u00e4rjestet\u00e4\u00e4n l\u00e4hiviikkoina ja jossa k\u00e4sitell\u00e4\u00e4n komissio tiedonantoa YKP:n kansainv\u00e4lisest\u00e4 ulottuvuudesta sen uudistuksessa. PARSEP Mietinn\u00f6n osalta toivomme, ett\u00e4 komission tiedonannossa tuetaan kolmansien maiden kalastusalueilla liikkuvan Euroopan unionin laivaston ongelmien ja tarpeiden kokonaisvaltaista k\u00e4sittely\u00e4. On keskitytt\u00e4v\u00e4 kahteen kysymykseen: miten p\u00e4\u00e4st\u00e4\u00e4n johdonmukaisempaan osallistumiseen kansainv\u00e4lisiss\u00e4 vesialueita hallinnoivissa j\u00e4rjest\u00f6iss\u00e4 unionin legitiimien etujen puolustamiseksi ja miten saadaan YKP:n kansainv\u00e4linen ulottuvuus keskeisemm\u00e4lle sijalle unionin ulkopolitiikassa. PARSEP Ei ole juurikaan j\u00e4rke\u00e4 ryhty\u00e4 sellaisiin toimiin kuin rakennetukien poistamiseen tai alusten romuttamisen tukemiseen YKP:n uudistamisen yhteydess\u00e4, jos sit\u00e4 ennen ei ole m\u00e4\u00e4ritetty unionin kantaa yhteis\u00f6n laivastoon. T\u00e4ten puollamme mietint\u00f6\u00e4 huolimatta toimista, jotka neuvosto on jo toteuttanut alalla. PARSEP Esityslista on nyt k\u00e4yty kokonaan l\u00e4pi. PARSEP Julistan Euroopan parlamentin istuntokauden keskeytetyksi. PARSEP (Istunto p\u00e4\u00e4ttyi klo 17.05.) PARSEP . (PT) Tied\u00e4mme kaikki, kuinka t\u00e4rke\u00e4\u00e4 on j\u00e4senvaltioiden alusten toiminta unionin ulkopuolisilla kalastusalueilla. Se on ratkaisevaa, jotta markkinoille saadaan riitt\u00e4v\u00e4sti kalatuotteita, joista on pulaa, ja jotta toimintaa yll\u00e4pidet\u00e4\u00e4n kalastuksesta riippuvaisilla unionin seuduilla, jotka ovat perinteisesti olleet ep\u00e4suotuisia ja vailla taloudellisen toiminnan vaihtoehtoja. PARSEP Olen \u00e4\u00e4nest\u00e4nyt mietinn\u00f6n puolesta korostaakseni asian merkitt\u00e4vyytt\u00e4 ja ilmaistakseni olevani samaa mielt\u00e4 esittelij\u00e4n tekem\u00e4st\u00e4 arviosta. Olen yht\u00e4 mielt\u00e4 monista h\u00e4nen ehdotuksistaan. Lainaan FAO:n vastuullisen kalastuksen toimintaohjeiden johdantoa (FAO:n yleiskokouksen 28. istunto, 31. lokakuuta 1995) \"Kalastus ja vesiviljely ovat t\u00e4rkeit\u00e4 ravinnontuotannon l\u00e4hteit\u00e4, ty\u00f6llist\u00e4ji\u00e4 ja vapaa-ajan harrastuksia, ne edist\u00e4v\u00e4t kaupank\u00e4ynti\u00e4 ja tarjoavat taloudellista hyvinvointia kaikille maailman ihmisille, nykyisille ja tuleville sukupolville, joten niit\u00e4 olisi harjoitettava vastuullisella tavalla.\" Haluaisin tuoda esille esittelij\u00e4n varoituksen, ett\u00e4 kaikkea laitonta kalastusta on torjuttava; kehotuksen, ett\u00e4 V\u00e4limeren maiden suhteita on tiivistett\u00e4v\u00e4, ja pyynn\u00f6n, ett\u00e4 yhteisyritysten perustamista kolmansien maiden kanssa olisi edistett\u00e4v\u00e4. PARSEP Komission on viel\u00e4 selvennett\u00e4v\u00e4, miten se aikoo varmistaa, ett\u00e4 yhteisen kalastuspolitiikan kansainv\u00e4linen ulottuvuus saadaan keskeisemm\u00e4lle sijalle unionin ulkopolitiikassa. On my\u00f6s ratkaisevan t\u00e4rke\u00e4\u00e4 s\u00e4ilytt\u00e4\u00e4 tehokas tuki kalastusalusten uusimiseksi. PARSEP";

            Document leftDocument = new Document();
            leftDocument.Add(new Field("body", body, ft));
            leftWriter.AddDocument(leftDocument);

            Document rightDocument = new Document();
            rightDocument.Add(new Field("body", body, ft));
            rightWriter.AddDocument(rightDocument);

            IndexReader leftReader = leftWriter.GetReader();
            leftWriter.Dispose();
            IndexReader rightReader = rightWriter.GetReader();
            rightWriter.Dispose();

            // check that our readers are valid
            TestUtil.CheckReader(leftReader);
            TestUtil.CheckReader(rightReader);

            string info = "left: " + leftCodec.ToString() + " / right: " + rightCodec.ToString();

            // From AssertReaderEquals

            var leftFields = MultiFields.GetFields(leftReader);
            var rightFields = MultiFields.GetFields(rightReader);

            // From AssertFieldsEquals

            using var leftEnum = leftFields.GetEnumerator();
            using var rightEnum = rightFields.GetEnumerator();

            while (leftEnum.MoveNext())
            {
                string field = leftEnum.Current;
                rightEnum.MoveNext();
                Assert.AreEqual(field, rightEnum.Current, info);

                var leftTerms = leftFields.GetTerms(field);
                var rightTerms = rightFields.GetTerms(field);

                // From AssertTermsEquals

                //string re = "??(*)+*.\U000e06d7*"; // Before escaping
                string re = "??(\ue808*)+*.\udb41\uded7*"; // Faulty Regex
                Util.Automaton.CompiledAutomaton automaton = new Util.Automaton.CompiledAutomaton((new Util.Automaton.RegExp(re, Util.Automaton.RegExpSyntax.NONE)).ToAutomaton());
                if (automaton.Type == Util.Automaton.CompiledAutomaton.AUTOMATON_TYPE.NORMAL)
                {
                    // From AssertTermsEnumEquals

                    BytesRef term;
                    TermsEnum leftTermsEnum = leftTerms.Intersect(automaton, null);
                    TermsEnum rightTermsEnum = rightTerms.Intersect(automaton, null);

                    while (leftTermsEnum.MoveNext())
                    {
                        term = leftTermsEnum.Term;
                        rightTermsEnum.MoveNext();
                        Assert.AreEqual(term, rightTermsEnum.Term, info);
                    }
                    Assert.IsFalse(rightTermsEnum.MoveNext(), info);
                }
            }
            Assert.IsFalse(rightEnum.MoveNext());

            Util.IOUtils.Dispose(leftReader, rightReader);
            Util.IOUtils.Dispose(leftDir, rightDir);
        }
    }
}