package org.apache.turbine.modules;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import org.apache.turbine.Turbine;
import org.apache.turbine.pipeline.PipelineData;

/**
 * The purpose of this class is to allow one to load and execute
 * Action modules.
 *
 * @author <a href="mailto:mbryson@mont.mindspring.com">Dave Bryson</a>
 * @author <a href="mailto:hps@intermeta.de">Henning P. Schmiedehausen</a>
 * @author <a href="mailto:peter@courcoux.biz">Peter Courcoux</a>
 * @version $Id: ActionLoader.java 1773378 2016-12-09 13:19:59Z tv $
 */
public class ActionLoader
    extends GenericLoader<Action>
    implements Loader<Action>
{
    /** The single instance of this class. */
    private static ActionLoader instance = new ActionLoader();

    /**
     * These ctor's are private to force clients to use getInstance()
     * to access this class.
     */
    private ActionLoader()
    {
        super();
    }

    /**
     * Attempts to load and execute the external action.
     *
     * @param pipelineData Turbine information.
     * @param name Name of object that will execute the action.
     * @throws Exception a generic exception.
     */
    @Override
    public void exec(PipelineData pipelineData, String name)
    		throws Exception
    {
        getAssembler(name).perform(pipelineData);
    }

    /**
     * Pulls out an instance of the object by name.  Name is just the
     * single name of the object.
     *
     * @param name Name of object instance.
     * @return An Action with the specified name, or null.
     * @throws Exception a generic exception.
     */
    @Override
    public Action getAssembler(String name)
        throws Exception
    {
        return getAssembler(Action.class, name);
    }

    /**
     * @see org.apache.turbine.modules.Loader#getCacheSize()
     */
    @Override
    public int getCacheSize()
    {
        return ActionLoader.getConfiguredCacheSize();
    }

    /**
     * The method through which this class is accessed.
     *
     * @return The single instance of this class.
     */
    public static ActionLoader getInstance()
    {
        return instance;
    }

    /**
     * Helper method to get the configured cache size for this module
     *
     * @return the configure cache size
     */
    private static int getConfiguredCacheSize()
    {
        return Turbine.getConfiguration().getInt(Action.CACHE_SIZE_KEY,
                Action.CACHE_SIZE_DEFAULT);
    }
}
